// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   Spherical
//-----------------------------------------------------------------------------

#ifndef POOMA_COORDINATESYSYEMS_SPHERICAL_H
#define POOMA_COORDINATESYSYEMS_SPHERICAL_H

//namespace pooma_spherical_coordinates {

//-----------------------------------------------------------------------------
// Overview: 
// 
// Spherical : A 3D spherical coordinate system (r,theta,phi)
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Region.h"
#include "Tiny/Vector.h"

//-----------------------------------------------------------------------------
// Full Description:
// 
// Spherical is a 3-dimensional cylindrical coordinate system.
//   Coordinate ordering is (r, phi, theta).
//   r     = radius
//   phi   = azimuthal angle, range (0,2*pi)
//   theta = polar angle, starting along positive z axis, range (0,pi)
//-----------------------------------------------------------------------------

class Spherical
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  typedef Spherical This_t;

  // The dimensionality of this coordinate system.
  enum { dimensions = 3 };

  //---------------------------------------------------------------------------
  // Constructors.

  Spherical() {
    oneThird_m = 1.0/3.0;
  }

  // Compute the volume of a 3D region, interpreted as a range of coordinate
  // values for each coordinate.

  template <class T>
  inline
  T volume(Region<3,T> &region)
  {
    T r1 = region[0].first();
    T r2 = region[0].last();
    return oneThird_m * (r2*r2*r2 - r1*r1*r1) * 
      (cos(region[1].first()) - cos(region[1].last)) * region[2].size();
  }

  // Compute the "volume" of a 3D region, interpreted as a range of coordinate
  // values for each coordinate. Here, we allow for zero-length in any of the
  // coordinates, and return an area or length when one or two coordinates of
  // the region have zero length:

  template <class T>
  inline
  T lengthAreaVolume(Region<3,T> &region)
  {
    T r1 = region[0].first();
    T r2 = region[0].last();
    T deltaR = region[0].size();
    T deltaPhi = region[1].size();
    T deltaTheta = region[2].size();
    if (deltaRho == 0.0) {
      // No extent radially
      if (deltaTheta == 0.0) {
	      // No extent in theta, meaning return arc length in phi
	      return r1 * deltaPhi;
      } else {
	      // Finite extent in theta
	      if (deltaPhi == 0.0) {
	        // No extent in phi; return arc length in theta
	        return r1 * deltaTheta;
	      } else {
	        // Finite extent in phi; return area in theta-phi
	        return r1*r1 * deltaPhi * deltaTheta;
	      }
      } 
    } else {
      // Finite radial extent
      if (deltaTheta == 0.0) {
	      // No extent in theta
	      if (deltaPhi == 0.0) {
	        // No extent in phi; return length in r
	        return deltaR;
	      } else {
	        // Finite extent in phi; return r-phi area
	        return (r2*r2 - r1*r1) * deltaPhi;
	      }
      } else {
	      // Finite extent in theta
	      if (deltaPhi == 0.0) {
	        // No extent in theta; return r-theta area
	        return (r2*r2 - r1*r1) * deltaTheta;
	      } else {
	        // Finite extent in z; return 3D volume
	        return oneThird_m * (r2*r2*r2 - r1*r1*r1) * 
            (cos(region[2].first()) - cos(region[2].last)) * region[1].size();
	      }
      }
    }
    // If we got here, it's an error; put in an error check:
  }

  // Volume of a general solid (dense set of points in space). Defer
  // calculation of volume to the class representing the solid.
  
  /*
  template<class Solid>
  typename Solid::PointComponentType_t volume(const Solid &solid) const
  {
    return solid.volume();
  }
  */

  // (Euclidean) distance between two points:
  template<class T>
  inline
  T distance(Vector<3, T> &pt1, Vector<3, T> &pt2)
  {
    T x1 = pt1(0) * sin(pt1(2)) * cos(pt1(1));
    T x2 = pt2(0) * sin(pt2(2)) * cos(pt2(1));
    T y1 = pt1(0) * sin(pt1(2)) * sin(pt1(1));
    T y2 = pt2(0) * sin(pt2(2)) * sin(pt2(1));
    T z1 = pt1(0) * cos(pt1(2));
    T z2 = pt2(0) * cos(pt2(2));
    return 
      sqrt((x2 - x1)*(x2 - x1) + (y2 - y1)*(y2 - y1) + (z2 - z1)*(z2 - z1));
  }

private:

  double oneThird_m;

};


//} // namespace pooma_spherical_coordinates

#endif // POOMA_COORDINATESYSYEMS_SPHERICAL_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Spherical.h,v $   $Author: julianc $
// $Revision: 1.3 $   $Date: 2000/07/25 00:26:07 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
