// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class: Evaluator
//----------------------------------------------------------------------------

#ifndef POOMA_EVALUATOR_EVALUATOR_H
#define POOMA_EVALUATOR_EVALUATOR_H

//-----------------------------------------------------------------------------
// Overview:
//
// Evaluator evaluates expressions by examining the engines that are
// participating in the expression and dispatching to custom code.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Evaluator/CompressibleEval.h"
#include "Evaluator/ExpressionKernel.h"
#include "Evaluator/EvaluatorTags.h"
#include "Engine/Intersector.h"
#include "Engine/IntersectEngine.h"
#include "Engine/NotifyEngineWrite.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
//
// The point of this class is to input an expression with the
// 'evaluate' member function and evaluate it by breaking it up into
// appropriate sub-blocks, looping over the whole domain, and
// evaluating the expression at each point.
//
//-----------------------------------------------------------------------------

template <class EvalTag>
struct Evaluator
{
};

//-----------------------------------------------------------------------------
// Main Evaluator:
//
// This evaluator is the one that gets called for a data-parallel expression.
// It just determines the appropriate evaluator from the types of the LHS and
// RHS.  Also, we block if appropriate.
//-----------------------------------------------------------------------------

template <>
struct Evaluator<MainEvaluatorTag>
{
  // Default ctor.

  Evaluator() { }

  // Destructor

  ~Evaluator() { }

  // evaluate(expression)
  // Input an expression and cause it to be evaluated.
  // We just pass the buck to a special evaluator.

  template <class LHS, class RHS, class Op>
  void evaluate(const LHS& lhs, const Op& op, const RHS& rhs) const
  {
    typedef typename EvaluatorTag<LHS, RHS>::Evaluator_t Eval_t;
    Evaluator<Eval_t> evaluator;

    Pooma::beginExpression();
    evaluator.evaluate(lhs(), op, rhs());
    notifyEngineWrite(lhs.engine());
    Pooma::endExpression();
    
    POOMA_INCREMENT_STATISTIC(NumExpressions)
  }

  // evaluateZeroBased(expression)
  // Input an expression and cause it to be evaluated.
  // We just pass the buck to a special evaluator.
  // This version does not bother to take views of the expression
  // since the caller is assuring us they are already zero-based.

  template <class LHS, class RHS, class Op>
  void evaluateZeroBased(const LHS& lhs, const Op& op, const RHS& rhs) const
  {
    typedef typename EvaluatorTag<LHS, RHS>::Evaluator_t Eval_t;
    Evaluator<Eval_t> evaluator;

    Pooma::beginExpression();
    evaluator.evaluate(lhs, op, rhs);
    notifyEngineWrite(lhs.engine());
    Pooma::endExpression();
    
    POOMA_INCREMENT_STATISTIC(NumZBExpressions)
  }
};


//-----------------------------------------------------------------------------
// Single-patch Evaluator:
//
// The single patch version just passes the tag on to generate
// an expression kernel.
//-----------------------------------------------------------------------------

template <>
struct Evaluator<SinglePatchEvaluatorTag>
{
  // Default ctor.

  Evaluator() { }

  // Destructor

  ~Evaluator() { }

  // evaluate(expression)
  // Input an expression and cause it to be evaluated.
  // We just pass the buck to a special evaluator.

  template <class LHS, class RHS, class Op>
  void evaluate(const LHS& lhs, const Op& op, const RHS& rhs) const
  {
    typedef typename KernelTag<LHS,RHS>::Kernel_t Kernel_t;
#if POOMA_REORDER_ITERATES
    Pooma::Iterate_t *iterate = generateKernel(lhs, op, rhs, Kernel_t());
    Pooma::scheduler().handOff(iterate);
#else
    KernelEvaluator<Kernel_t>::evaluate(lhs, op, rhs);
#endif
  }
};


//-----------------------------------------------------------------------------
// Multiple-patch Evaluator:
//
// The multiple patch version makes patches and sends them out to
// the single patch evaluator.
//-----------------------------------------------------------------------------

template <>
struct Evaluator<MultiPatchEvaluatorTag>
{
  // Default ctor.

  Evaluator() { }

  // Destructor

  ~Evaluator() { }

  // evaluate(expression)
  // Input an expression and cause it to be evaluated.
  // We just pass the buck to a special evaluator.

  template <class LHS, class RHS, class Op>
  void evaluate(const LHS& lhs, const Op& op, const RHS& rhs) const
  {
    typedef Intersector<LHS::dimensions> Inter_t;
    Inter_t inter;
    EngineApply<IntersectorTag<Inter_t> > tag(inter);
    
    forEach(lhs, tag, NullCombine());
    forEach(rhs, tag, NullCombine());
  
    Inter_t::const_iterator i = inter.begin();
    while (i != inter.end())
    {
      Evaluator<SinglePatchEvaluatorTag>().evaluate(lhs(*i), op, rhs(*i));
      ++i;
    }
    
    POOMA_INCREMENT_STATISTIC(NumMultiPatchExpressions)
    POOMA_INCREMENT_STATISTIC_BY(NumLocalPatchesEvaluated, inter.size())
  }
};


#endif // POOMA_EVALUATOR_EVALUATOR_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Evaluator.h,v $   $Author: sa_smith $
// $Revision: 1.55 $   $Date: 2000/07/14 00:39:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
