// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
// 
// Div
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_DIFFOPS_DIV_RH_H
#define POOMA_FIELD_DIFFOPS_DIV_RH_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// Classes:
//
// Div	:  Divergence operator on Fields, using 2nd-order centered differences
//         These are used by the div() template function. See Div.h for
//         details, and the div() function template definition.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Field/DiffOps/Div.h"
#include "Geometry/DiscreteGeometry.RM.h"
#include "Tiny/Vector.h"
#include "Tiny/Tensor.h"
#include "Tiny/VectorTensor.h"
#include "Tiny/VectorOperators.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
// 
// Classes:
//
// Div:
// 
// Partial specializations of the generic Div (divergence) functor. See Div.h
// for general comments.  These are for RectilinearMesh-based Geometry.
// 
// Div is a functor class serving as the "Functor" template parameter for
// FieldStencil<Functor>. Its operator() functions implement second-order 
// centered differences on an input Field and return output elements of an 
// output Field. The types of the input and output Field differ in two ways:
//	1) The input centering is (possibly) different than the output 
//	   centering.
//	2) The input element type is Vector<Dim,T2> (or Tensor<Dim,T2>)  
//         and the output type is a scalar type T2 (or Vector<Dim,T2>).
// Partial specializations implement various combinations of input and output
// centerings, for specific coordinate systems.
// 
// Exported typedefs:
//
// OutputCentering_t - Centering parameter of the output DiscreteGeometry class
// OutputElement_t   - Type of the elements in the output ConstField; 
//                     restricted to a scalar type (vector input) or 
//                     vector (tensor input)
// 
// Accessors:
//
// lowerExtent(int d) - Returns the stencil width in direction d, at the "low"
//                      end of the (logically) rectilinear mesh. This is the
//                      maximim positive integer offset from the element 
//                      indexed by integer i in the input Field's index space
//                      along dimension d used in outputting the element
//                      indexed by integer i in the output Field's index space
//                      along dimension d
// upperExtent(int d) - Same as lowerExtent(), but for the "high" end of the 
//                      mesh. That is, the maximum (magnitude) *negative* 
//                      offset from i in direction d.
// 
// Other methods:
// 
// operator()(...) - The actual implementation for the stencil. This acts on 
//              a set of scalar-indexed values in the input Field's 
//              index space making up the stencil, as offset from the 
//              fixed index point specified by
//              the function's input arguments (list of scalar index values). 
//              The stencil must be written so that the same fixed index point
//              specified by the input arguments where the values are to be
//              assigned in the index space of the output Field. This means, 
//              for example, that if the operator is going from one centering 
//              to a different output centering, the index bookkeeping must 
//              be done correctly by this operator()() function implementation.
//-----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Partial specializations of Div
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Divergence Vector/Vert -> Scalar/Cell
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Div<Cell, 
  DiscreteGeometry<Vert, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  Vector<Dim, T2> >
{
public:

  typedef Cell OutputCentering_t;
  typedef T2 OutputElement_t;

  // 
  // Constructors.
  // 

  Div()
  {
    T2 coef = 1.0;
    int d;
    for (d = 1; d < Dim; d++) { coef *= 0.5; }
    for (d = 0; d < Dim; d++) {
      for (int b = 0; b < (1 << Dim); b++) {
        int s = ( b & (1 << d) ) ? 1 : -1;
        Dvc_m[b](d) = s*coef;
      }
    }
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 0; }
  int upperExtent(int d) const { return 1; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    return (dot(f.read(i1    ), 
                Dvc_m[0]/f.geometry().mesh().vertexDeltas().read(i1)) +
	    dot(f.read(i1 + 1), 
                Dvc_m[1]/f.geometry().mesh().vertexDeltas().read(i1)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return (dot(f.read(i1    , i2    ), Dvc_m[0]/vD.read(i1, i2)) +
	    dot(f.read(i1 + 1, i2    ), Dvc_m[1]/vD.read(i1, i2)) +
	    dot(f.read(i1    , i2 + 1), Dvc_m[2]/vD.read(i1, i2)) +
	    dot(f.read(i1 + 1, i2 + 1), Dvc_m[3]/vD.read(i1, i2)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return (dot(f.read(i1    , i2    , i3    ), Dvc_m[0]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1 + 1, i2    , i3    ), Dvc_m[1]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1    , i2 + 1, i3    ), Dvc_m[2]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1 + 1, i2 + 1, i3    ), Dvc_m[3]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1    , i2    , i3 + 1), Dvc_m[4]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1 + 1, i2    , i3 + 1), Dvc_m[5]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1    , i2 + 1, i3 + 1), Dvc_m[6]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1 + 1, i2 + 1, i3 + 1), Dvc_m[7]/vD.read(i1, i2, i3)));
  }

private:

  // Geometrical constants for derivatives:
  Vector<Dim,T2> Dvc_m[1<<Dim];
};

// ----------------------------------------------------------------------------
// Divergence Vector/Cell -> Scalar/Vert
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Div<Vert, 
  DiscreteGeometry<Cell, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  Vector<Dim, T2> >
{
public:

  typedef Vert OutputCentering_t;
  typedef T2 OutputElement_t;

  // 
  // Constructors.
  // 

  Div()
  {
    T2 coef = 1.0;
    int d;
    for (d = 1; d < Dim; d++) { coef *= 0.5; }
    for (d = 0; d < Dim; d++) {
      for (int b = 0; b < (1 << Dim); b++) {
        int s = ( b & (1 << d) ) ? 1 : -1;
        Dvc_m[b](d) = s*coef;
      }
    }
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }
  int upperExtent(int d) const { return 0; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return (dot(f.read(i1-1), Dvc_m[0]/(cellX.read(i1) - cellX.read(i1-1))) +
	    dot(f.read(i1  ), Dvc_m[1]/(cellX.read(i1) - cellX.read(i1-1))));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return 
      (dot(f.read(i1-1, i2-1), 
           Dvc_m[0]/(cellX.read(i1, i2) - cellX.read(i1-1, i2-1))) +
       dot(f.read(i1  , i2-1), 
           Dvc_m[1]/(cellX.read(i1, i2) - cellX.read(i1-1, i2-1))) +
       dot(f.read(i1-1, i2  ), 
           Dvc_m[2]/(cellX.read(i1, i2) - cellX.read(i1-1, i2-1))) +
       dot(f.read(i1  , i2  ), 
           Dvc_m[3]/(cellX.read(i1, i2) - cellX.read(i1-1, i2-1))));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return 
      (dot(f.read(i1-1, i2-1, i3-1), 
	   Dvc_m[0]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1  , i2-1, i3-1), 
	   Dvc_m[1]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1-1, i2  , i3-1), 
	   Dvc_m[2]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1  , i2  , i3-1), 
	   Dvc_m[3]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1-1, i2-1, i3  ), 
	   Dvc_m[4]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1  , i2-1, i3  ), 
	   Dvc_m[5]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1-1, i2  , i3  ), 
	   Dvc_m[6]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1  , i2  , i3  ), 
	   Dvc_m[7]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))));
  }

private:

  // Geometrical constants for derivatives:
  Vector<Dim,T2> Dvc_m[1<<Dim];
};

// ----------------------------------------------------------------------------
// Divergence Vector/Cell -> Scalar/Cell
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Div<Cell, 
  DiscreteGeometry<Cell, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  Vector<Dim, T2> >
{
public:

  typedef Cell OutputCentering_t;
  typedef T2 OutputElement_t;

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }

  int upperExtent(int d) const { return 1; }
   
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return ((f.comp(0).read(i1 + 1) - f.comp(0).read(i1 - 1)) / 
	    (cellX.comp(0).read(i1 + 1) - cellX.comp(0).read(i1 - 1)));
  }


  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return ((f.comp(0).read(i1 + 1, i2) - f.comp(0).read(i1 - 1, i2)) / 
	    (cellX.comp(0).read(i1 + 1, i2) - cellX.comp(0).read(i1 - 1, i2)) +
	    (f.comp(1).read(i1, i2 + 1) - f.comp(1).read(i1, i2 - 1)) / 
	    (cellX.comp(1).read(i1, i2 + 1) - cellX.comp(1).read(i1, i2 - 1)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return ((f.comp(0).read(i1 + 1, i2, i3) - 
             f.comp(0).read(i1 - 1, i2, i3)) / 
	    (cellX.comp(0).read(i1 + 1, i2, i3) - 
             cellX.comp(0).read(i1 - 1, i2, i3)) +
	    (f.comp(1).read(i1, i2 + 1, i3) - 
             f.comp(1).read(i1, i2 - 1, i3)) / 
	    (cellX.comp(1).read(i1, i2 + 1, i3) - 
             cellX.comp(1).read(i1, i2 - 1, i3)) +
	    (f.comp(2).read(i1, i2, i3 + 1) - 
             f.comp(2).read(i1, i2, i3 - 1)) / 
	    (cellX.comp(2).read(i1, i2, i3 + 1) - 
             cellX.comp(2).read(i1, i2, i3 - 1)));

  }

};

// ----------------------------------------------------------------------------
// Divergence Vector/Vert -> Scalar/Vert
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Div<Vert, 
  DiscreteGeometry<Vert, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  Vector<Dim, T2> >
{
public:

  typedef Vert OutputCentering_t;
  typedef T2 OutputElement_t;

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }

  int upperExtent(int d) const { return 1; }
   
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return ((f.comp(0).read(i1 + 1) - f.comp(0).read(i1 - 1)) / 
	    (vD.comp(0).read(i1 - 1) + vD.comp(0).read(i1)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return ((f.comp(0).read(i1 + 1, i2) - f.comp(0).read(i1 - 1, i2)) / 
	    (vD.comp(0).read(i1 - 1, i2) + vD.comp(0).read(i1, i2)) +
	    (f.comp(1).read(i1, i2 + 1) - f.comp(1).read(i1, i2 - 1)) / 
	    (vD.comp(1).read(i1, i2 - 1) + vD.comp(1).read(i1, i2)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return ((f.comp(0).read(i1 + 1, i2, i3) - 
             f.comp(0).read(i1 - 1, i2, i3)) / 
	    (vD.comp(0).read(i1 - 1, i2, i3) + vD.comp(0).read(i1, i2, i3)) +
	    (f.comp(1).read(i1, i2 + 1, i3) - 
             f.comp(1).read(i1, i2 - 1, i3)) / 
	    (vD.comp(1).read(i1, i2 - 1, i3) + vD.comp(1).read(i1, i2, i3)) +
	    (f.comp(2).read(i1, i2, i3 + 1) - 
             f.comp(2).read(i1, i2, i3 - 1)) / 
	    (vD.comp(2).read(i1, i2, i3 - 1) + vD.comp(2).read(i1, i2, i3)));
  }

};

// ----------------------------------------------------------------------------
// Divergence Tensor/Vert -> Vector/Cell
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2, class TensorEngineTag>
class Div<Cell, 
  DiscreteGeometry<Vert, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  Tensor<Dim, T2, TensorEngineTag> >
{
public:

  typedef Cell OutputCentering_t;
  typedef Vector<Dim, T2> OutputElement_t;

  // 
  // Constructors.
  // 

  Div()
  {
    T2 coef = 1.0;
    int d;
    for (d = 1; d < Dim; d++) { coef *= 0.5; }
    for (d = 0; d < Dim; d++) {
      for (int b = 0; b < (1 << Dim); b++) {
        int s = ( b & (1 << d) ) ? 1 : -1;
        Dvc_m[b](d) = s*coef;
      }
    }
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 0; }
  int upperExtent(int d) const { return 1; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return (dot(f.read(i1    ), Dvc_m[0]/vD.read(i1)) + 
	    dot(f.read(i1 + 1), Dvc_m[1]/vD.read(i1)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return (dot(f.read(i1    , i2    ), Dvc_m[0]/vD.read(i1, i2)) +
	    dot(f.read(i1 + 1, i2    ), Dvc_m[1]/vD.read(i1, i2)) +
	    dot(f.read(i1    , i2 + 1), Dvc_m[2]/vD.read(i1, i2)) +
	    dot(f.read(i1 + 1, i2 + 1), Dvc_m[3]/vD.read(i1, i2)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return (dot(f.read(i1    , i2    , i3    ), Dvc_m[0]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1 + 1, i2    , i3    ), Dvc_m[1]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1    , i2 + 1, i3    ), Dvc_m[2]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1 + 1, i2 + 1, i3    ), Dvc_m[3]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1    , i2    , i3 + 1), Dvc_m[4]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1 + 1, i2    , i3 + 1), Dvc_m[5]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1    , i2 + 1, i3 + 1), Dvc_m[6]/vD.read(i1, i2, i3)) +
	    dot(f.read(i1 + 1, i2 + 1, i3 + 1), Dvc_m[7]/vD.read(i1, i2, i3)));
  }

private:

  // Geometrical constants for derivatives:
  Vector<Dim,T2> Dvc_m[1<<Dim];
};


// ----------------------------------------------------------------------------
// Divergence Tensor/Cell -> Vector/Vert
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2, class TensorEngineTag>
class Div<Vert, 
  DiscreteGeometry<Cell, RectilinearMesh<Dim, Cartesian<Dim>, T1 > >,
  Tensor<Dim, T2, TensorEngineTag> >
{
public:

  typedef Vert OutputCentering_t;
  typedef Vector<Dim, T2> OutputElement_t;

  // 
  // Constructors.
  // 

  Div()
  {
    T2 coef = 1.0;
    int d;
    for (d = 1; d < Dim; d++) { coef *= 0.5; }
    for (d = 0; d < Dim; d++) {
      for (int b = 0; b < (1 << Dim); b++) {
        int s = ( b & (1 << d) ) ? 1 : -1;
        Dvc_m[b](d) = s*coef;
      }
    }
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }
  int upperExtent(int d) const { return 0; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return (dot(f.read(i1-1), Dvc_m[0]/(cellX.read(i1) - cellX.read(i1-1))) +
	    dot(f.read(i1  ), Dvc_m[1]/(cellX.read(i1) - cellX.read(i1-1))));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return 
      (dot(f.read(i1-1, i2-1), 
           Dvc_m[0]/(cellX.read(i1, i2) - cellX.read(i1-1, i2-1))) +
       dot(f.read(i1  , i2-1), 
           Dvc_m[1]/(cellX.read(i1, i2) - cellX.read(i1-1, i2-1))) +
       dot(f.read(i1-1, i2  ), 
           Dvc_m[2]/(cellX.read(i1, i2) - cellX.read(i1-1, i2-1))) +
       dot(f.read(i1  , i2  ), 
           Dvc_m[3]/(cellX.read(i1, i2) - cellX.read(i1-1, i2-1))));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return 
      (dot(f.read(i1-1, i2-1, i3-1), 
	   Dvc_m[0]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1  , i2-1, i3-1), 
	   Dvc_m[1]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1-1, i2  , i3-1), 
	   Dvc_m[2]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1  , i2  , i3-1), 
	   Dvc_m[3]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1-1, i2-1, i3  ), 
	   Dvc_m[4]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1  , i2-1, i3  ), 
	   Dvc_m[5]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1-1, i2  , i3  ), 
	   Dvc_m[6]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       dot(f.read(i1  , i2  , i3  ), 
	   Dvc_m[7]/(cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))));
  }

private:

  // Geometrical constants for derivatives:
  Vector<Dim,T2> Dvc_m[1<<Dim];
};

#endif     // POOMA_FIELD_DIFFOPS_DIV_RH_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Div.RM.h,v $   $Author: julianc $
// $Revision: 1.14 $   $Date: 2000/07/21 23:07:09 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
