// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
// 
// Grad
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_DIFFOPS_GRAD_RM_H
#define POOMA_FIELD_DIFFOPS_GRAD_RM_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// Classes:
//
// Grad	:  Gradient operator on Fields, using 2nd-order centered differences
//         These are used by the grad() template function. See Grad.h for
//         details, and the grad() function template definition.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Field/DiffOps/Grad.h"
#include "Geometry/DiscreteGeometry.RM.h"
#include "Tiny/Vector.h"
#include "Tiny/Tensor.h"
#include "Tiny/VectorTensor.h"
#include "Tiny/VectorOperators.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
// 
// Classes:
//
// Grad:
// 
// Partial specializations of the generic Grad (gradient) functor. See Grad.h
// for general comments. These are for RectilinearMesh-based DiscreteGeometry.
// 
// Grad is a functor class serving as the "Functor" template parameter for
// FieldStencil<Functor>. Its operator() functions implement 2nd 
// order centered differences on an input Field and return output elements 
// of an output Field.  The types of the input and output Field differ 
// in two ways:
//	1) The input centering is (possibly) different than the output 
//	   centering.
//	2) The input element type is a scalar of type T2 (or a Vector<Dim,T2>)
//         and the output type is a Vector<Dim,T2> (or Tensor<Dim,T2>).
// Partial specializations implement various combinations of input and output
// centerings, for specific coordinate systems.
// 
// Exported typedefs:
//
// OutputCentering_t - Centering parameter of the output DiscreteGeometry class
// OutputElement_t   - Type of the elements in the output ConstField; 
//                     restricted to a scalar type (vector input) or vector
//                     (tensor input)
// 
// Accessors:
//
// lowerExtent(int d) - Returns the stencil width in direction d, at the "low"
//                      end of the (logically) rectilinear mesh. This is the
//                      maximim positive integer offset from the element 
//                      indexed by integer i in the input Field's index space
//                      along dimension d used in outputting the element
//                      indexed by integer i in the output Field's index space
//                      along dimension d
// upperExtent(int d) - Same as lowerExtent(), but for the "high" end of the 
//                      mesh. That is, the maximum (magnitude) *negative*
//                      offset from i in direction d.
// 
// Other methods:
// 
// operator()(...) - The actual implementation for the stencil. This acts on a
//                   set of scalar-indexed values in the input Field's index
//                   space making up the stencil, as offset from the fixed
//                   index point specified by the function's input arguments
//                   (list of scalar index values).  The stencil must be
//                   written so that the same fixed index point specified by
//                   the input arguments where the values are to be assigned in
//                   the index space of the output Field. This means, for
//                   example, that if the operator is going from one centering
//                   to a different output centering, the index bookkeeping
//                   must be done correctly by this operator()() function
//                   implementation.
//-----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Partial specializations of Grad
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Gradient Scalar/Vert -> Vector/Cell
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Grad<Cell, 
  DiscreteGeometry<Vert, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  T2>
{
public:

  typedef Cell OutputCentering_t;
  typedef Vector<Dim, T2> OutputElement_t;

  // 
  // Constructors.
  // 

  Grad()
  {
    T2 coef = 1.0;
    int d;
    for (d = 1; d < Dim; d++) { coef *= 0.5; }
    for (d = 0; d < Dim; d++) {
      for (int b = 0; b < (1 << Dim); b++) {
        int s = ( b & (1 << d) ) ? 1 : -1;
        Dvc_m[b](d) = s*coef;
      }
    }
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 0; }
  int upperExtent(int d) const { return 1; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    return 
      OutputElement_t((f.read(i1  ) * Dvc_m[0] + f.read(i1+1) * Dvc_m[1]) / 
		      f.geometry().mesh().vertexDeltas().read(i1));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    return (f.read(i1    , i2    ) * Dvc_m[0] +
	    f.read(i1 + 1, i2    ) * Dvc_m[1] +
	    f.read(i1    , i2 + 1) * Dvc_m[2] +
	    f.read(i1 + 1, i2 + 1) * Dvc_m[3]) / 
      f.geometry().mesh().vertexDeltas().read(i1, i2);
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    return (f.read(i1    , i2    , i3    ) * Dvc_m[0] +
	    f.read(i1 + 1, i2    , i3    ) * Dvc_m[1] +
	    f.read(i1    , i2 + 1, i3    ) * Dvc_m[2] +
	    f.read(i1 + 1, i2 + 1, i3    ) * Dvc_m[3] +
	    f.read(i1    , i2    , i3 + 1) * Dvc_m[4] +
	    f.read(i1 + 1, i2    , i3 + 1) * Dvc_m[5] +
	    f.read(i1    , i2 + 1, i3 + 1) * Dvc_m[6] +
	    f.read(i1 + 1, i2 + 1, i3 + 1) * Dvc_m[7])/ 
      f.geometry().mesh().vertexDeltas().read(i1, i2, i3);
  }

private:

  // Geometrical constants for derivatives:
  Vector<Dim,T2> Dvc_m[1<<Dim];
};

// ----------------------------------------------------------------------------
// Gradient Scalar/Cell -> Vector/Vert
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Grad<Vert, 
  DiscreteGeometry<Cell, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  T2>
{
public:

  typedef Vert OutputCentering_t;
  typedef Vector<Dim, T2> OutputElement_t;

  // 
  // Constructors.
  // 

  Grad()
  {
    T2 coef = 1.0;
    int d;
    for (d = 1; d < Dim; d++) { coef *= 0.5; }
    for (d = 0; d < Dim; d++) {
      for (int b = 0; b < (1 << Dim); b++) {
        int s = ( b & (1 << d) ) ? 1 : -1;
        Dvc_m[b](d) = s*coef;
      }
    }
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }
  int upperExtent(int d) const { return 0; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return (f.read(i1 - 1) * Dvc_m[0] +
	    f.read(i1    ) * Dvc_m[1]) / 
            (cellX.read(i1) - cellX.read(i1-1));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return 
      (f.read(i1 - 1, i2 - 1) * Dvc_m[0] +
       f.read(i1    , i2 - 1) * Dvc_m[1] +
       f.read(i1 - 1, i2    ) * Dvc_m[2] +
       f.read(i1    , i2    ) * Dvc_m[3]) / 
       (cellX.read(i1, i2) - cellX.read(i1-1, i2-1));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return (f.read(i1 - 1, i2 - 1, i3 - 1) * Dvc_m[0] +
	    f.read(i1    , i2 - 1, i3 - 1) * Dvc_m[1] +
	    f.read(i1 - 1, i2    , i3 - 1) * Dvc_m[2] +
	    f.read(i1    , i2    , i3 - 1) * Dvc_m[3] +
	    f.read(i1 - 1, i2 - 1, i3    ) * Dvc_m[4] +
	    f.read(i1    , i2 - 1, i3    ) * Dvc_m[5] +
	    f.read(i1 - 1, i2    , i3    ) * Dvc_m[6] +
	    f.read(i1    , i2    , i3    ) * Dvc_m[7]) / 
            (cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1));
  }

private:

  // Geometrical constants for derivatives:
  Vector<Dim,T2> Dvc_m[1<<Dim];
};

// ----------------------------------------------------------------------------
// Gradient Scalar/Vert -> Vector/Vert
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Grad<Vert, 
  DiscreteGeometry<Vert, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  T2>
{
public:

  typedef Vert OutputCentering_t;
  typedef Vector<Dim, T2> OutputElement_t;

  typedef Grad<Vert, 
  DiscreteGeometry<Vert, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  T2> This_t;

  Grad() { }
  Grad(const This_t &) { }
  This_t &operator=(const This_t &) { return *this; }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }
  int upperExtent(int d) const { return 1; }
   
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return OutputElement_t((f.read(i1 + 1) - f.read(i1 - 1)) / 
			   (vD.comp(0).read(i1) + vD.comp(0).read(i1 - 1)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return 
      OutputElement_t((f.read(i1 + 1, i2    ) - f.read(i1 - 1, i2    )) / 
		      (vD.comp(0).read(i1, i2) + vD.comp(0).read(i1 - 1, i2)), 
		      (f.read(i1    , i2 + 1) - f.read(i1    , i2 - 1)) / 
		      (vD.comp(1).read(i1, i2) + vD.comp(1).read(i1, i2 - 1)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return 
      OutputElement_t((f.read(i1 + 1, i2    , i3    ) - 
                       f.read(i1 - 1, i2    , i3    )) /
		      (vD.comp(0).read(i1, i2, i3) + 
                       vD.comp(0).read(i1 - 1, i2, i3)), 
		      (f.read(i1    , i2 + 1, i3    ) - 
                       f.read(i1    , i2 - 1, i3    )) /
		      (vD.comp(1).read(i1, i2, i3) + 
                       vD.comp(1).read(i1, i2 - 1, i3)),
		      (f.read(i1    , i2    , i3 + 1) - 
                       f.read(i1    , i2    , i3 - 1)) /
		      (vD.comp(2).read(i1, i2, i3) + 
                       vD.comp(2).read(i1, i2, i3 - 1)));
  }

};

// ----------------------------------------------------------------------------
// Gradient Scalar/Cell -> Vector/Cell
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Grad<Cell, 
  DiscreteGeometry<Cell, RectilinearMesh<Dim, Cartesian<Dim>, T1 > >,
  T2>
{
public:

  typedef Cell OutputCentering_t;
  typedef Vector<Dim, T2> OutputElement_t;

  typedef Grad<Cell, 
  DiscreteGeometry<Cell, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  T2> This_t;

  Grad() { }
  Grad(const This_t &) { }
  This_t &operator=(const This_t &) { return *this; }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }
  int upperExtent(int d) const { return 1; }
   
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return 
      OutputElement_t((f.read(i1 + 1) - f.read(i1 - 1)) /
		      (cellX.comp(0).read(i1+1) - cellX.comp(0).read(i1-1)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return 
      OutputElement_t((f.read(i1 + 1, i2    ) - f.read(i1 - 1, i2    )) /
		      (cellX.comp(0).read(i1+1, i2) - 
                       cellX.comp(0).read(i1-1, i2)), 
		      (f.read(i1    , i2 + 1) - f.read(i1    , i2 - 1)) /
		      (cellX.comp(1).read(i1, i2+1) - 
                       cellX.comp(1).read(i1,i2-1)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return 
      OutputElement_t((f.read(i1 + 1, i2    , i3    ) - 
                       f.read(i1 - 1, i2    , i3    )) /
		      (cellX.comp(0).read(i1+1, i2, i3) - 
		       cellX.comp(0).read(i1-1, i2, i3)),
		      (f.read(i1    , i2 + 1, i3    ) - 
                       f.read(i1    , i2 - 1, i3    )) /
		      (cellX.comp(1).read(i1, i2+1, i3) - 
		       cellX.comp(1).read(i1, i2-1, i3)),
		      (f.read(i1    , i2    , i3 + 1) - 
                       f.read(i1    , i2    , i3 - 1)) /
		      (cellX.comp(2).read(i1, i2, i3+1) - 
		       cellX.comp(2).read(i1, i2, i3-1)));
  }

};


// ----------------------------------------------------------------------------
// Gradient Vector/Vert -> Tensor/Cell
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Grad<Cell, 
  DiscreteGeometry<Vert, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  Vector<Dim, T2> >
{
public:

  typedef Cell OutputCentering_t;
  typedef Tensor<Dim, T2> OutputElement_t;

  // 
  // Constructors.
  // 

  Grad()
  {
    T2 coef = 1.0;
    int d;
    for (d = 1; d < Dim; d++) { coef *= 0.5; }
    for (d = 0; d < Dim; d++) {
      for (int b = 0; b < (1 << Dim); b++) {
        int s = ( b & (1 << d) ) ? 1 : -1;
        Dvc_m[b](d) = s*coef;
      }
    }
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 0; }
  int upperExtent(int d) const { return 1; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return (outerProduct(f.read(i1    ), Dvc_m[0] / vD.read(i1)) +
	    outerProduct(f.read(i1 + 1), Dvc_m[1] / vD.read(i1)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return (outerProduct(f.read(i1    , i2    ), Dvc_m[0] / vD.read(i1, i2)) +
	    outerProduct(f.read(i1 + 1, i2    ), Dvc_m[1] / vD.read(i1, i2)) +
	    outerProduct(f.read(i1    , i2 + 1), Dvc_m[2] / vD.read(i1, i2)) +
	    outerProduct(f.read(i1 + 1, i2 + 1), Dvc_m[3] / vD.read(i1, i2)));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::Mesh_t::SpacingsArray_t &vD = 
      f.geometry().mesh().vertexDeltas();
    return 
      (outerProduct(f.read(i1    , i2    , i3    ), 
                    Dvc_m[0] / vD.read(i1, i2, i3)) +
       outerProduct(f.read(i1 + 1, i2    , i3    ), 
                    Dvc_m[1] / vD.read(i1, i2, i3)) +
       outerProduct(f.read(i1    , i2 + 1, i3    ), 
                    Dvc_m[2] / vD.read(i1, i2, i3)) +
       outerProduct(f.read(i1 + 1, i2 + 1, i3    ), 
                    Dvc_m[3] / vD.read(i1, i2, i3)) +
       outerProduct(f.read(i1    , i2    , i3 + 1), 
                    Dvc_m[4] / vD.read(i1, i2, i3)) +
       outerProduct(f.read(i1 + 1, i2    , i3 + 1), 
                    Dvc_m[5] / vD.read(i1, i2, i3)) +
       outerProduct(f.read(i1    , i2 + 1, i3 + 1), 
                    Dvc_m[6] / vD.read(i1, i2, i3)) +
       outerProduct(f.read(i1 + 1, i2 + 1, i3 + 1), 
                    Dvc_m[7] / vD.read(i1, i2, i3)));
  }

private:

  // Geometrical constants for derivatives:
  Vector<Dim,T2> Dvc_m[1<<Dim];
};


// ----------------------------------------------------------------------------
// Gradient Vector/Cell -> Tensor/Vert
// ----------------------------------------------------------------------------

template<int Dim, class T1, class T2>
class Grad<Vert, 
  DiscreteGeometry<Cell, RectilinearMesh<Dim, Cartesian<Dim>, T1> >,
  Vector<Dim, T2> >
{
public:

  typedef Vert OutputCentering_t;
  typedef Tensor<Dim, T2> OutputElement_t;

  // 
  // Constructors.
  // 

  Grad()
  {
    T2 coef = 1.0;
    int d;
    for (d = 1; d < Dim; d++) { coef *= 0.5; }
    for (d = 0; d < Dim; d++) {
      for (int b = 0; b < (1 << Dim); b++) {
        int s = ( b & (1 << d) ) ? 1 : -1;
        Dvc_m[b](d) = s*coef;
      }
    }
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }
  int upperExtent(int d) const { return 0; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return (outerProduct(f.read(i1 - 1), 
			 Dvc_m[0] / (cellX.read(i1) - cellX.read(i1-1))) +
	    outerProduct(f.read(i1    ), 
			 Dvc_m[1] / (cellX.read(i1) - cellX.read(i1-1))));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return (outerProduct(f.read(i1 - 1, i2 - 1), 
			 Dvc_m[0] / 
                         (cellX.read(i1, i2) - cellX.read(i1-1, i2-1))) +
	    outerProduct(f.read(i1    , i2 - 1), 
			 Dvc_m[1] / 
                         (cellX.read(i1, i2) - cellX.read(i1-1, i2-1))) +
	    outerProduct(f.read(i1 - 1, i2    ), 
			 Dvc_m[2] / 
                         (cellX.read(i1, i2) - cellX.read(i1-1, i2-1))) +
	    outerProduct(f.read(i1    , i2    ), 
			 Dvc_m[3] / 
                         (cellX.read(i1, i2) - cellX.read(i1-1, i2-1))));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    const typename F::Geometry_t::PositionArray_t &cellX = f.geometry().x();
    return 
      (outerProduct(f.read(i1 - 1, i2 - 1, i3 - 1), 
		    Dvc_m[0] / 
                    (cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       outerProduct(f.read(i1    , i2 - 1, i3 - 1), 
		    Dvc_m[1] / 
                    (cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       outerProduct(f.read(i1 - 1, i2    , i3 - 1), 
		    Dvc_m[2] / 
                    (cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       outerProduct(f.read(i1    , i2    , i3 - 1), 
		    Dvc_m[3] / 
                    (cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       outerProduct(f.read(i1 - 1, i2 - 1, i3    ), 
		    Dvc_m[4] / 
                    (cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       outerProduct(f.read(i1    , i2 - 1, i3    ), 
		    Dvc_m[5] / 
                    (cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       outerProduct(f.read(i1 - 1, i2    , i3    ), 
		    Dvc_m[6] / 
                    (cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))) +
       outerProduct(f.read(i1    , i2    , i3    ), 
		    Dvc_m[7] / 
                    (cellX.read(i1, i2, i3) - cellX.read(i1-1, i2-1, i3-1))));
  }

private:

  // Geometrical constants for derivatives:
  Vector<Dim,T2> Dvc_m[1<<Dim];
};

#endif     // POOMA_FIELD_DIFFOPS_GRAD_RM_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Grad.RM.h,v $   $Author: julianc $
// $Revision: 1.13 $   $Date: 2000/07/21 23:07:09 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
