// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Functions: 
//   sum             - sum all the elements in an array.
//   prod            - multiply all of the elements in an array.
//   max             - find the maximum value in an array.
//   min             - find the minimum value in an array.
//   all             - returns true if all of the array's elements are != 0.
//   any             - returns true if any of the array's elements are != 0.
//   bitOr           - does a bitwise or of all of the elements.
//   bitAnd          - does a bitwise and of all of the elements.
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_REDUCTIONS_H
#define POOMA_FIELD_REDUCTIONS_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include "Evaluator/Reduction.h"


//-----------------------------------------------------------------------------
// Forward definitions
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
class ConstField;


//-----------------------------------------------------------------------------
// Specific global reduction functions for ConstFields.
//-----------------------------------------------------------------------------

// Sum up the elements of a ConstField.

template<class Geom, class T, class EngineTag>
T sum(const ConstField<Geom, T, EngineTag> &f)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpAddAssign(), f);
  return ret;
}

// Compute the product of the elements of a ConstField.

template<class Geom, class T, class EngineTag>
T prod(const ConstField<Geom, T, EngineTag> &f)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpMultiplyAssign(), f);
  return ret;
}

// Find the smallest element of a ConstField.

template<class Geom, class T, class EngineTag>
T min(const ConstField<Geom, T, EngineTag> &f)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnMinAssign(), f);
  return ret;
}

// Find the largest element of a ConstField.

template<class Geom, class T, class EngineTag>
T max(const ConstField<Geom, T, EngineTag> &f)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnMaxAssign(), f);
  return ret;
}

// Report if all of the elments of the ConstField are true.

template<class Geom, class T, class EngineTag>
bool all(const ConstField<Geom, T, EngineTag> &f)
{
  bool ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnAndAssign(), f);
  return ret;
}

// Report if some of the elments of the ConstField are true.

template<class Geom, class T, class EngineTag>
bool any(const ConstField<Geom, T, EngineTag> &f)
{
  bool ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnOrAssign(), f);
  return ret;
}

// Bitwise-or all of the elements together.

template<class Geom, class T, class EngineTag>
T bitOr(const ConstField<Geom, T, EngineTag> &f)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpBitwiseOrAssign(), f);
  return ret;
}

// Bitwise-and all of the elements together.

template<class Geom, class T, class EngineTag>
T bitAnd(const ConstField<Geom, T, EngineTag> &f)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpBitwiseAndAssign(), f);
  return ret;
}

#endif // POOMA_FIELD_REDUCTIONS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Reductions.h,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/03/28 23:25:48 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
