// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   Lagrangian<Dim, TM, CoordinateSystem>
//   FieldEngine<Lagrangian<...>, T, EngineTag>
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELDENGINE_LAGRANGIAN_H
#define POOMA_NEWFIELD_FIELDENGINE_LAGRANGIAN_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// FieldEngine<Lagrangian, T, EngineTag>
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "NewField/FieldEngine/FieldEngineBase.h"

#include "Array/Array.h"
#include "CoordinateSystems/Cartesian.h"
#include "Domain/Interval.h"
#include "Engine/ConstantFunctionEngine.h"
#include "Engine/EnginePatch.h"
#include "Engine/IndexFunctionEngine.h"
#include "Layout/INode.h"
#include "PETE/PETE.h"
#include "Pooma/Indices.h"
#include "Tiny/Vector.h"
#include "Utilities/PAssert.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag> class FieldEngine;


//-----------------------------------------------------------------------------
// Full Description:
// 
// Lagrangian tag class.
//-----------------------------------------------------------------------------

template<int Dim, class TM =  POOMA_DEFAULT_POSITION_TYPE,
  class CoordinateSystem = Cartesian<Dim> >
struct Lagrangian;


//-----------------------------------------------------------------------------
// Full Description:
// 
// FieldEngine, specialized for a rudimentary Lagrangian mesh.
//-----------------------------------------------------------------------------

template<int Dim, class TM, class CoordinateSystem, class T, class EngineTag>
class FieldEngine<Lagrangian<Dim, TM, CoordinateSystem>, T, EngineTag> :
  public FieldEngineBase<Dim, T, EngineTag>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  
  typedef FieldEngine<Lagrangian<Dim, TM, CoordinateSystem>, T, EngineTag>
    This_t;
    
  // Our base class.
  
  typedef FieldEngineBase<Dim, T, EngineTag> Base_t;

  // The geometry tag.
  
  typedef Lagrangian<Dim, TM, CoordinateSystem> GeometryTag_t;
  
  // The engine tag.
  
  typedef EngineTag EngineTag_t;
  
  // The coordinate system.
  
  typedef CoordinateSystem CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename Base_t::Domain_t Domain_t;
  
  // The number of indices required to select a point in this mesh.
  
  enum { dimensions = Dim };
  
  // The type used to represent a point in the mesh.
  
  typedef Vector<Dim, TM> PointType_t;
  
  // Our engine type.
  
  typedef Engine<Dim, T, EngineTag> Engine_t;
  
  // Our layout type.
  
  typedef typename Engine_t::Layout_t Layout_t;


  //---------------------------------------------------------------------------
  // Constructors. We don't use default arguments for the origin and spacings
  // because that produces ambiguities.

  template<class BaseInitializer>  
  FieldEngine(const BaseInitializer &baseInit, 
    const Layout_t &layout,
    const PointType_t &origin,
    const PointType_t &spacings)
  : Base_t(baseInit, layout),
    positions_m(layout)
    {
      positions_m = origin + iota(growRight(totalCellDomain(), 1) - 
        physicalCellDomain().firsts()) * spacings;
    }
    
  // Copy constructor.
  
  FieldEngine(const This_t &model)
  : Base_t(model),
    positions_m(model.positions_m)
    { }
    
  // Sub-field constructor.
  
  FieldEngine(const This_t &model, const int &iSubField)
  : Base_t(model, iSubField),
    positions_m(model.positions_m)
    { }

  // Domain view constructors.

  template<class T2, class ET2>  
  FieldEngine(const FieldEngine<GeometryTag_t, T2, ET2> &model, 
    const Domain_t &d)
  : Base_t(model, d),
    positions_m(model.vertexPositions()(translateToVertexDomain(d)))
    { }

  template<class T2, class ET2>  
  FieldEngine(const FieldEngine<GeometryTag_t, T2, ET2> &model, 
    const INode<Dim> &inode)
  : Base_t(model, inode),
    positions_m(model.vertexPositions()(inode))
    { }

  // Expression-engine constructor.    

  FieldEngine(const Engine<Dim, T, EngineTag> &e)
  : Base_t(e),
    positions_m(referenceField().fieldEngine().vertexPositions())
    { }

  // Patch constructor.    

  template<class ET2>
  FieldEngine(const FieldEngine<GeometryTag_t, T, ET2> &model, 
    const EnginePatch &p)
  : Base_t(model, p),
    positions_m(model.vertexPositions().patchLocal(p.patch_m))
    { }


  //---------------------------------------------------------------------------
  // Copy assignment operator (shallow).
  
  This_t &operator=(const This_t &rhs);


  //---------------------------------------------------------------------------
  // Empty destructor is fine.
  
  ~FieldEngine() { }


  //---------------------------------------------------------------------------
  // Return the vertex centers. 
  
  typedef Array<Dim, PointType_t, EngineTag_t> VertexPositionsArray_t;

  const VertexPositionsArray_t &vertexPositions() const
    {
      return positions_m;
    }
  
private:

  VertexPositionsArray_t positions_m;

};

template<int Dim, class TM, class CoordinateSystem, class T, class EngineTag,
  class Tag>
struct LeafFunctor<
  FieldEngine<Lagrangian<Dim, TM, CoordinateSystem>, T, EngineTag>, 
  EngineApply<Tag> >
{
  typedef Lagrangian<Dim, TM, CoordinateSystem> GeometryTag_t;
  typedef FieldEngine<GeometryTag_t, T, EngineTag> Subject_t;
  typedef typename Subject_t::Engine_t Engine_t;
  typedef LeafFunctor<Engine_t, EngineApply<Tag> > LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t Type_t;

  inline static
  Type_t apply(const Subject_t &fieldEngine, const EngineApply<Tag> &tag)
  {
    forEach(fieldEngine.vertexPositions(), tag, NullCombine());
    
    return LeafFunctor_t::apply(fieldEngine.engine(), tag);
  }
};

#endif // POOMA_NEWFIELD_FIELDENGINE_LAGRANGIAN_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldEngine.Lagrangian.h,v $   $Author: swhaney $
// $Revision: 1.2 $   $Date: 2000/07/20 21:03:51 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

