// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   FieldEngineBase
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELDENGINE_FIELDENGINEBASE_H
#define POOMA_NEWFIELD_FIELDENGINE_FIELDENGINEBASE_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// FieldEngineBase and related classes. POOMA supports a flexible form 
// of "centering" that allows a hierarchy of multiple centering points per 
// cell. The centering information, managemed by the FieldEngineBase
// class, is initialized using a flexible set of functors.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Interval.h"
#include "Domain/Loc.h"
#include "Domain/Shrink.h"
#include "Layout/INode.h"
#include "Utilities/PAssert.h"
#include "Utilities/RefCountedBlockPtr.h"

#include "NewField/Updater/UpdaterList.h"


//-----------------------------------------------------------------------------
// Forward declarations:
//-----------------------------------------------------------------------------

template<int Dim, class T, class EngineTag> class Engine;
template<class Components> class ComponentWrapper;


// ----------------------------------------------------------------------------
// FieldEngineBaseData holds offset information about the centering along
// with a copy of the engine containg the data associated with the centering.
// ----------------------------------------------------------------------------

template <int Dim, class T, class EngineTag>
class FieldEngineBaseData : public RefCounted
{
public:

  template<class Initializer>
  FieldEngineBaseData(const Loc<Dim> &offsets, 
                      const Initializer &init)
    : offsets_m(offsets), 
      engine_m(init)
    { }

  template<class Initializer>
  FieldEngineBaseData(const Loc<Dim> &offsets, 
                      const Initializer &init, const UpdaterList &l)
    : offsets_m(offsets), 
      engine_m(init),
      updaters_m(l)
    { }

  template<class Engine, class Domain>
  FieldEngineBaseData(const Loc<Dim> &offsets, const Engine &e,
    const Domain &d, const UpdaterList &l)
    : offsets_m(offsets), 
      engine_m(NewEngineEngine<Engine, Domain>::apply(e, d), 
               NewEngineDomain<Engine, Domain>::apply(e, d)),
      updaters_m(l)
    { }
    
  const Loc<Dim> &offsets() const { return offsets_m; }

  const Engine<Dim, T, EngineTag> &engine() const { return engine_m; }
  Engine<Dim, T, EngineTag> &engine() { return engine_m; }

  const UpdaterList &updaters() const { return updaters_m; }
  UpdaterList &updaters() { return updaters_m; }
  
private:

  Loc<Dim> offsets_m;
  Engine<Dim, T, EngineTag> engine_m;
  UpdaterList updaters_m;
};


// ----------------------------------------------------------------------------
// FieldEngineBase manages a hierarchy of engines, making it possible for
// FieldEngine specializations to implement geometry-specific behavior only.
// ----------------------------------------------------------------------------

template<int Dim, class T, class EngineTag>
class FieldEngineBase
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.
    
  typedef FieldEngineBase<Dim, T, EngineTag> This_t;
  typedef FieldEngineBaseData<Dim, T, EngineTag> Data_t;
  typedef Engine<Dim, T, EngineTag> Engine_t;
  typedef typename Engine_t::Domain_t Domain_t;
  typedef typename Engine_t::Layout_t Layout_t;
  typedef typename Engine_t::Element_t Element_t;
  typedef typename Engine_t::ElementRef_t ElementRef_t;
  enum { dimensions = Dim };
  typedef GuardLayers<Dim> GuardLayers_t;


  //---------------------------------------------------------------------------
  // Constructors.

  // Default constructor.
  
  FieldEngineBase()
  : physicalCellDomain_m(Pooma::NoInit()),
    guards_m(0)
    { }
  
  // Layout constructor. Takes a layout appropriate for the contained
  // engines and, based on the logic contained in the InitFunctor, sets
  // up centering information and the engines.
    
  template<class InitFunctor>
  FieldEngineBase(const InitFunctor &f, const Layout_t &layout)
  : physicalCellDomain_m(layout.domain()),
    guards_m(layout.externalGuards())
    {
      shrinkInPlace(physicalCellDomain_m, guards_m);
      shrinkRightInPlace(physicalCellDomain_m, 1);
      f.initialize(*this, layout);
    }

  // Initialize with a new centering from a model.

  template<class InitFunctor, class T2, class ET2>
  FieldEngineBase(const InitFunctor &f,
		  const FieldEngineBase<Dim, T2, ET2> &model)
  : physicalCellDomain_m(model.physicalCellDomain()),
    guards_m(model.guardLayers())
    {
      f.initialize(*this, Pooma::NoInit());
    }

  // Copy constructor.
  
  FieldEngineBase(const This_t &model)
  : subFields_m(model.subFields_m),
    data_m(model.data_m),
    physicalCellDomain_m(model.physicalCellDomain_m),
    guards_m(model.guards_m)
    { }

  // Sub-field view constructor. This is when we want to construct a view of
  // one of the subFields in our top-level list.
  
  FieldEngineBase(const This_t &model, const int &iSubField)
  : subFields_m(model.subFields_m[iSubField].subFields_m),
    data_m(model.subFields_m[iSubField].data_m),
    physicalCellDomain_m(model.physicalCellDomain_m),
    guards_m(model.guards_m)
    { }  

  // View constructor.  
  
  template<int Dim2, class T2, class EngineTag2, class Initializer>
  FieldEngineBase(const FieldEngineBase<Dim2, T2, EngineTag2> &model, 
    const Initializer &i)
    {
      initialize(*this, model, i); 
    }  

  //---------------------------------------------------------------------------
  // Initialize functions. 

  template<class Initializer>
  void initialize(const Loc<Dim> &loc, const Initializer &init)
    {
      data_m = new Data_t(loc, init);
    }

  template<class Initializer>
  void initialize(const Loc<Dim> &loc, const Initializer &init, 
                  const UpdaterList &l)
    {
      data_m = new Data_t(loc, init, l);
    }

  template<class Engine, class Domain>
  void initialize(const Loc<Dim> &loc, const Engine &e, const Domain &d,
    const UpdaterList &l)
    {
      data_m = new Data_t(loc, e, d, l);
    }

  void initialize(const This_t &model)
  {
    subFields_m = model.subFields_m;
    data_m = model.data_m;
    physicalCellDomain_m = model.physicalCellDomain_m;
    guards_m = model.guards_m;
  }

  template<int Dim2, class T2, class EngineTag2>
  void initialize(This_t &s, 
    const FieldEngineBase<Dim2, T2, EngineTag2> &model, const Domain_t &d)
    {
      int n = model.numSubFields();
      if (n == 0)
        {
          s.physicalCellDomain_m = shrinkRight(d - d.firsts(),
					       model.offsets());
          s.initialize(model.offsets(), model.engine(), d, model.updaters());
        }
      else
        {
          s.physicalCellDomain_m = d - d.firsts();
          s.addSubFields(n);
          for (int i = 0; i < n; i++)
            initialize(s.subFields_m[i], model.subField(i),
              model.subField(i).translateDomain(d));
        }
    }

  template<int Dim2, class T2, class EngineTag2>
  void initialize(This_t &s, 
    const FieldEngineBase<Dim2, T2, EngineTag2> &model, const INode<Dim> &i)
    {
      PAssert(model.numSubFields() == 0);
      s.physicalCellDomain_m = 
        shrinkRight(i.domain() - i.domain().firsts(), model.offsets());
      s.initialize(model.offsets(), model.engine(), i, model.updaters());
    }

  template<class EngineTag2>
  void initialize(This_t &s, 
    const FieldEngineBase<Dim, T, EngineTag2> &model, const EnginePatch &p)
    {
      PAssert(model.numSubFields() == 0);
      s.initialize(model.offsets(), engineFunctor(model.engine(), p));
      s.physicalCellDomain_m = shrinkRight(data_m->engine().domain(), 1);
    }

  template<int Dim2, class T2, class EngineTag2, class Components>
  void initialize(This_t &s, 
    const FieldEngineBase<Dim2, T2, EngineTag2> &model, 
    const ComponentWrapper<Components> &c)
    {
      int n = model.numSubFields();
      if (n == 0)
        {
          s.physicalCellDomain_m = model.physicalCellDomain();
          s.guards_m = model.guardLayers();
          s.initialize(model.offsets(), 
                       Engine_t(model.engine(), c.components()),
                       model.updaters());
        }
      else
        {
          s.physicalCellDomain_m = model.physicalCellDomain();
          s.guards_m = model.guardLayers();
          s.addSubFields(n);
          for (int i = 0; i < n; i++)
            initialize(s.subFields_m[i], model.subField(i), c);
        }
    }
      

  //---------------------------------------------------------------------------
  // Accessors and modifiers.
    
  void addSubFields(int n)
    {
      PAssert(subFields_m.size() == 0);
      
      subFields_m.reserve(n);
      subFields_m.resize(n);
      for (int i = 0; i < n; i++)
        {
          subFields_m[i].physicalCellDomain_m = physicalCellDomain_m;
          subFields_m[i].guards_m = guards_m;
        }
    }
    
  inline int numSubFields() const
    {
      return subFields_m.size();
    }
    
  This_t &subField(int i)
    {
      return subFields_m[i];
    }
    
  const This_t &subField(int i) const
    {
      return subFields_m[i];
    }

  const Loc<Dim> &offsets() const
    {
      PAssert(data_m.isValid());
      return data_m->offsets();
    }
    
  Engine_t &engine()
    {
      PAssert(data_m.isValid());
      return data_m->engine();
    }
    
  const Engine_t &engine() const
    {
      PAssert(data_m.isValid());
      return data_m->engine();
    }

  const UpdaterList &updaters() const
    {
      PAssert(data_m.isValid());
      return data_m->updaters();
    }

  UpdaterList &updaters()
    {
      PAssert(data_m.isValid());
      return data_m->updaters();
    }
    
  const GuardLayers_t &guardLayers() const
    {
      return guards_m;
    }

  GuardLayers_t &guardLayers()
    {
      return guards_m;
    }


  //---------------------------------------------------------------------------
  // Domain accessor functions. 
        
  inline const Domain_t &physicalCellDomain() const
    {
      return physicalCellDomain_m;
    }
        
  inline Domain_t totalCellDomain() const
    {
      return grow(physicalCellDomain_m, guards_m);
    }

  Domain_t physicalDomain(int iSubField) const
    {
      return subFields_m[iSubField].
        translateDomain(physicalCellDomain_m);
    }

  Domain_t totalDomain(int iSubField) const
    {
      return subFields_m[iSubField].
        translateDomain(totalCellDomain());
    }

  Domain_t physicalDomain() const
    {
      return translateDomain(physicalCellDomain_m);
    }

  Domain_t totalDomain() const
    {
      return translateDomain(totalCellDomain());
    }

  
  //---------------------------------------------------------------------------
  // Make a distinct copy of this fieldEngineBase.   
 
  template<class Subject>
  void makeOwnCopy(const Subject &s)
    {
      // This only make sense if we have no subfields.
      
      PAssert(data_m.isValid());

      // Create a new data block with shallow copies of the engine and the
      // updaters.
      
      initialize(offsets(), engine(), updaters());
      
      // Deepen the copies of the engine & updaters list.
      
      engine().makeOwnCopy();
      updaters().makeOwnCopy(s);
    }


  //---------------------------------------------------------------------------
  // Domain translation function. 
  
  inline Domain_t
  translateDomain(const Domain_t &d) const
    {
      if (subFields_m.size() == 0)
        return growRight(d, offsets());
      else
        return d;
    }
  
  inline Domain_t
  translateToVertexDomain(const Domain_t &d) const
    {
      if (subFields_m.size() == 0)
        return d;
      else
        return growRight(d, 1);
    }

      
private:

  RefCountedBlockPtr<This_t> subFields_m;
  RefCountedPtr<Data_t> data_m;
  Domain_t physicalCellDomain_m;
  GuardLayers_t guards_m;
};

#endif // POOMA_NEWFIELD_FIELDENGINE_FIELDENGINEBASE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldEngineBase.h,v $   $Author: sa_smith $
// $Revision: 1.5 $   $Date: 2000/07/26 16:51:38 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
