// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// non-inline definitions for PatchSizeSyncer
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Tulip/PatchSizeSyncer.h"
#include "Tulip/RemoteProxy.h"

#include <algorithm>

namespace Pooma {

int PatchSizeSyncer::tag_s = 0;

#if POOMA_CHEETAH
Cheetah::MatchingHandler *PatchSizeSyncer::handler_s = 0;
#endif

//-----------------------------------------------------------------------------
// PatchSize constructor & destructor...
//-----------------------------------------------------------------------------

PatchSizeSyncer::PatchSizeSyncer(int contextKey, Grid_t &localGrid)
  : myContext_m(Pooma::context()),
    numContexts_m(Pooma::contexts()),
    localKey_m(contextKey),
    localGrid_m(localGrid)
{ 
  if (myContext_m == 0) gridList_m.reserve(numContexts_m);
}

PatchSizeSyncer::~PatchSizeSyncer()
{
  PAssert(myContext_m == 0 || gridList_m.size() == 0);
  for (int i = 0; i < gridList_m.size(); ++i)
    delete gridList_m[i].second;
}

//-----------------------------------------------------------------------------
// PatchSizeSyncer::receiveGrid
//
// This function is passed on to the matching-handler and is invoked when
// a message is received.
//-----------------------------------------------------------------------------

void PatchSizeSyncer::receiveGrid(std::pair<int,Grid_t> &incoming)
{
  gridList_m.push_back(
    std::make_pair(incoming.first,new Grid_t(incoming.second)));
}

//-----------------------------------------------------------------------------
// PatchSizeSyncer::calcGlobalGrid
//
// Does a reduction of the grids, sending all the local grids to
// context 0, re-normalizing them to form a global grid, and broadcasting
// this global grid back to all contexts.
//-----------------------------------------------------------------------------

namespace {

// Functor used to sort the grid list after context 0 receives
// the remote grids.

struct ElemCompare
{
  typedef std::pair<int,Grid<1>*> Elem_t;
  inline bool operator()(const Elem_t &l, const Elem_t &r)
  {
    return l.first < r.first;
  }
};

}

void PatchSizeSyncer::calcGlobalGrid(Grid_t &globalGrid)
{
#if POOMA_CHEETAH

  // Each context will send their local Grid to context 0.
  // We'll offset the base tag by the context number - 1 to
  // generate the tags for this.

  int tagbase = tag_s;
  tag_s += numContexts_m - 1;

  Grid<1> result;

  if (myContext_m != 0)
    {
      handler_s->send(0, tagbase + myContext_m - 1, 
                      std::make_pair(localKey_m,localGrid_m));
    }
  else
    {
      // Push the context 0 grid onto the list:

      gridList_m.push_back(std::make_pair(localKey_m,new Grid_t(localGrid_m)));
      
      // Request messages from the other contexts, which 
      // will result in receiveGrid being invoked and 
      // the remainder of gridList_m being filled.
			  
      for (int i = 1; i < numContexts_m; ++i)
	{
	  handler_s->request(i, tagbase + i - 1, 
			     &PatchSizeSyncer::receiveGrid, 
			     this);
	}

      while (gridList_m.size() < numContexts_m)
	Pooma::poll();

      // The grid list is full. We sort it and then renormalize the
      // domains to make them globally consistent.  The
      // renormalization is done by looking through the list and
      // correcting each grid by adding to it the number of elements
      // have been added on the previous grids. We simultaneously
      // calculate the total number of points, needed to form the
      // global result.

      std::sort(gridList_m.begin(),gridList_m.end(),ElemCompare());

      int total_points = gridList_m[0].second->size() - 1;

      for (int j = 1; j < numContexts_m; ++j)
	{
	  Grid<1> &l = *gridList_m[j-1].second;
	  Grid<1> &r = *gridList_m[j].second;
	  r += l.last() - r.first();
	  total_points += r.size() - 1;
	}

      ++total_points; // for the final endpoint.

      // Finally, construct a composite Grid representing the global layout.

      IndirectionList<int> glist(total_points);

      int k = 0;
      for (int j = 0; j < numContexts_m; ++j)
	for (int i = 0; i < gridList_m[j].second->size() - 1; ++i)
	  glist(k++) = (*gridList_m[j].second)(i);

      glist(k) = gridList_m[numContexts_m-1].second->last();
      result = Grid<1>(glist);
    }

  // Broadcast the result.

  RemoteProxy<Grid<1> > broadcast(result,0);
  globalGrid = Grid<1>(broadcast.value());

#else  // POOMA_CHEETAH

  globalGrid = localGrid_m;

#endif // POOMA_CHEETAH
}


} // namespace Pooma


// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: PatchSizeSyncer.cmpl.cpp,v $   $Author: sa_smith $
// $Revision: 1.5 $   $Date: 2000/06/08 22:16:59 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
