/*****************************************************************************
* $CAMITK_LICENCE_BEGIN$
*
* CamiTK - Computer Assisted Medical Intervention ToolKit
* (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
*
* Visit http://camitk.imag.fr for more information
*
* This file is part of CamiTK.
*
* CamiTK is free software: you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License version 3
* only, as published by the Free Software Foundation.
*
* CamiTK is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU Lesser General Public License version 3 for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
*
* $CAMITK_LICENCE_END$
****************************************************************************/

#include "MeshComponent.h"
#include "InteractiveViewer.h"
#include "Property.h"
#include <Application.h>

//-- Qt stuff
#include <QMessageBox>
#include <QTextStream>
#include <QFileInfo>
#include <QTableView>
#include <QBoxLayout>
#include <QToolBar>
#include <QList>

//-- vtk stuff
#include <vtkPointSet.h>
#include <vtkUnstructuredGrid.h>
#include <vtkTetra.h>
#include <vtkHexahedron.h>
#include <vtkWedge.h>
#include <vtkPyramid.h>
#include <vtkCellArray.h>
#include <vtkGenericCell.h>
#include <vtkSelection.h>
#include <vtkSelectionNode.h>
#include <vtkCellData.h>
#include <vtkPointData.h>
#include <vtkDataSetAttributes.h>
#include <vtkExtractSelection.h>
#include <vtkDataSetMapper.h>
#include <vtkProperty.h>
#include <vtkHedgeHog.h>
#include <vtkTensorGlyph.h>
#include <vtkActor.h>
#include <vtkSphereSource.h>
#include <vtkCellCenters.h>

namespace camitk {

// -------------------- constructor --------------------
MeshComponent::MeshComponent(const QString & file) throw (AbortException) : Component(file, "Mesh", Component::GEOMETRY) {
    init();
}

MeshComponent::MeshComponent(vtkSmartPointer<vtkPointSet> aPointSet, const QString & name) : Component("", name, Component::GEOMETRY) {
    init();
    initRepresentation(aPointSet);
    setModified();
}

MeshComponent::MeshComponent(Component * parentComponent, vtkSmartPointer<vtkPointSet> aPointSet, const QString & name) : Component(parentComponent, name, Component::GEOMETRY) {
    init();
    initRepresentation(aPointSet);
}

// -------------------- destructor --------------------
MeshComponent::~MeshComponent() {
    if (getPointSet() != NULL)
        setActiveData(MeshDataModel::POINTS, NULL);

    if (selectionWidget != NULL)
        delete selectionWidget;

    if (dataWidget != NULL)
        delete dataWidget;
}

void MeshComponent::init() {
    pickedCellId = -1;
    pickedPointId = -1;

    // selection widget
    selectionWidget = new QWidget();
    selectionWidget->setObjectName("Selection");

    // selection actions
    removeSelections = new QAction(QPixmap(":/delete"), tr("Remove selection(s)"), this);
    removeSelections->setStatusTip(tr("Remove the selected selections"));
    removeSelections->setWhatsThis(tr("Remove the selected selections"));
    connect(removeSelections, SIGNAL(triggered()), this, SLOT(removeSelectedSelections()));

    inspectSelection = new QAction(QPixmap(":/settings"), tr("Inspect the selection"), this);
    inspectSelection->setStatusTip(tr("Inspect the selection"));
    inspectSelection->setWhatsThis(tr("Inspect the selection"));

    mergeSelection = new QAction(QPixmap(":/refresh"), tr("Merge the selections"), this);
    mergeSelection->setStatusTip(tr("Merge the selected selection"));
    mergeSelection->setWhatsThis(tr("Merge the selected delections (selection types must be identical)"));

    // selection model
    selectionModel = new MeshSelectionModel(this);

    // selection view
    selectionView = new MeshSelectionView(selectionWidget);
    selectionView->setSelectionBehavior(QAbstractItemView::SelectRows);
    selectionView->setModel(selectionModel);
    selectionView->setSelectionMode(QAbstractItemView::ExtendedSelection);
    selectionView->setEditTriggers(QAbstractItemView::DoubleClicked);

    policyBox = new QComboBox(selectionView);
    policyBox->setStatusTip(tr("Insertion policy"));
    policyBox->setWhatsThis(tr("Insertion policy"));
    policyBox->addItem("Replace", MeshSelectionModel::REPLACE);
    policyBox->addItem("Merge", MeshSelectionModel::MERGE);
    policyBox->addItem("Substract", MeshSelectionModel::SUBSTRACT);
    policyBox->addItem("Discard", MeshSelectionModel::DISCARD);

    // build the selection widget
    QVBoxLayout * hbox = new QVBoxLayout(selectionWidget);
    QToolBar * selectionToolBar = new QToolBar(selectionWidget);
    selectionToolBar->addAction(removeSelections);
    selectionToolBar->addAction(mergeSelection);
    selectionToolBar->addAction(inspectSelection);
    selectionToolBar->addWidget(policyBox);
    hbox->addWidget(selectionView);
    hbox->addWidget(selectionToolBar);
    selectionWidget->setLayout(hbox);

    // data widget
    dataWidget = new QWidget();
    dataWidget->setObjectName("Data");

    // selection actions
    removeData = new QAction(QPixmap(":/delete"), tr("Remove data"), this);
    removeData->setStatusTip(tr("Remove the selected data"));
    removeData->setWhatsThis(tr("Remove the selected data"));
    connect(removeData, SIGNAL(triggered()), this, SLOT(removeSelectedData()));

    inspectData = new QAction(QPixmap(":/settings"), tr("Inspect data"), this);
    inspectData->setStatusTip(tr("Inspect data"));
    inspectData->setWhatsThis(tr("Inspect data"));

    // data model
    dataModel = new MeshDataModel(this);

    // data view
    dataView = new MeshDataView();
    dataView->setSelectionBehavior(QAbstractItemView::SelectRows);
    dataView->setModel(dataModel);
    dataView->setSelectionMode(QAbstractItemView::ExtendedSelection);

    // build the data widget
    QVBoxLayout * dataBox = new QVBoxLayout(dataWidget);
    QToolBar * dataToolBar = new QToolBar(dataWidget);
    dataToolBar->addAction(removeData);
    dataToolBar->addAction(inspectData);
    dataBox->addWidget(dataView);
    dataBox->addWidget(dataToolBar);
    dataWidget->setLayout(dataBox);

    connect(selectionView->selectionModel(), SIGNAL(selectionChanged(const QItemSelection &, const QItemSelection &)), this, SLOT(changeSelectedSelection(const QItemSelection &, const QItemSelection &)));

    // selection
    currentSelection = vtkSmartPointer<vtkSelection>::New();
}

// -------------------- initRepresentation --------------------
void MeshComponent::initRepresentation(vtkSmartPointer<vtkPointSet> originalPointSet) {
    // if there is no point set yet, just do nothing
    if (originalPointSet == NULL)
        return;

    // else replace the point set
    if (myGeometry) {
        myGeometry->setPointSet(originalPointSet);
    } else {
        myGeometry = new Geometry(this->getName(), originalPointSet);
    }

    myGeometry->setMeshWorldTransform(getTransformFromWorld());

    // add it in the InteractiveViewer (automatically)
    setVisibility(InteractiveViewer::get3DViewer(), true);

    // initialize selection
    initSelection();

    // initialize data
    initData();

    // initialize dynamic properties
    initDynamicProperties();
}

// -------------------- initSelection --------------------
void MeshComponent::initSelection() {
    vtkSmartPointer<vtkExtractSelection> selectionExtractor = vtkSmartPointer<vtkExtractSelection>::New();
    vtkSmartPointer<vtkActor> selectionActor = vtkSmartPointer<vtkActor>::New();
    vtkSmartPointer<vtkDataSetMapper> selectionMapper = vtkSmartPointer<vtkDataSetMapper>::New();

    selectionExtractor->SetInputConnection(0, this->getDataPort());
    selectionExtractor->SetInputData(1, currentSelection);

    selectionMapper->SetInputConnection(selectionExtractor->GetOutputPort());

    selectionActor->SetPickable(false);
    selectionActor->GetProperty()->SetRepresentationToSurface();
    selectionActor->GetProperty()->SetLineWidth(5);
    selectionActor->GetProperty()->SetColor(1, 0, 0);
    selectionActor->GetProperty()->SetPointSize(10);
    selectionActor->GetProperty()->SetOpacity(0.2);
    selectionActor->SetMapper(selectionMapper);

    addProp("Selection", selectionActor);
}

// -------------------- initData --------------------
void MeshComponent::initData() {
    dataModel->refresh();
}

// -------------------- cellPicked --------------------
void MeshComponent::cellPicked(vtkIdType cellId, bool) {
    pickedCellId = cellId;
}

// -------------------- pointPicked --------------------
void MeshComponent::pointPicked(vtkIdType pointId, bool) {
    pickedPointId = pointId;
}

// -------------------- getPickedCellId --------------------
vtkIdType MeshComponent::getPickedCellId() {
    return pickedCellId;
}

// -------------------- getPickedPointId --------------------
vtkIdType MeshComponent::getPickedPointId() {
    return pickedPointId;
}

// -------------------- initDynamicProperties --------------------
void MeshComponent::initDynamicProperties() {
    vtkIdType count = 0;

    if (getPointSet() != NULL)
        count = getPointSet()->GetNumberOfPoints();

    Property * nbPoints = new Property("Number Of Points", QVariant(QString("%1").arg(count)), "Number of 3D Points composing the geometry", "");
    nbPoints->setReadOnly(true);
    addProperty(nbPoints);

    if (getPointSet() != NULL)
        count = getPointSet()->GetNumberOfCells();

    Property * nbCells = new Property("Number Of Cells", QVariant(QString("%1").arg(count)), "Number of Cells composing the geometry", "");
    nbCells->setReadOnly(true);
    addProperty(nbCells);

    if (getPointSet() != NULL) {
        // add a dynamic property to manage the surface color
        // setProperty("position point #1", QVector3D(1.0,0.0,0.0));
        vtkSmartPointer<vtkGenericCell> cell = vtkGenericCell::New();
        std::map<unsigned char, int> elementsMap;
        std::map<unsigned char, int>::iterator elementsMapIt;

        for (int i = 0; i < getPointSet()->GetNumberOfCells(); i++) {
            getPointSet()->GetCell(i, cell);

            if (!elementsMap.count(cell->GetCellType()))
                elementsMap[ cell->GetCellType()] = 0;

            elementsMap[ cell->GetCellType() ]++;

        }

        // the list of all possible cell types is defined in VTKCellType enum of the VTKCellType class
        for (elementsMapIt = elementsMap.begin(); elementsMapIt != elementsMap.end(); elementsMapIt++) {
            Property * cellProp;

            switch (elementsMapIt->first) {
            case VTK_EMPTY_CELL:
                cellProp = new Property("Empty Cells", elementsMapIt->second, tr("Number Of Empty Cells"), "");
                break;

            case VTK_VERTEX:
                cellProp = new Property("Vertex", elementsMapIt->second, tr("Number Of Vertex Cells"), "");
                break;

            case VTK_POLY_VERTEX:
                cellProp = new Property("Edges", elementsMapIt->second, tr("Number Of Edge Cells"), "");
                break;

            case VTK_LINE :
                cellProp = new Property("Lines", elementsMapIt->second, tr("Number Of Line Cells"), "");
                break;

            case VTK_POLY_LINE:
                cellProp = new Property("Polylines", elementsMapIt->second, tr("Number Of Polylines Cells"), "");
                break;

            case VTK_TRIANGLE :
                cellProp = new Property("Triangles", elementsMapIt->second, tr("Number Of Triangle Cells"), "");
                break;

            case VTK_TRIANGLE_STRIP:
                cellProp = new Property("Triangle Strips", elementsMapIt->second, tr("Number Of Triangle Strip Cells"), "");
                break;

            case VTK_POLYGON:
                cellProp = new Property("Polygons", elementsMapIt->second, tr("Number Of Polygon Cells"), "");
                break;

            case VTK_PIXEL:
                cellProp = new Property("Pixels", elementsMapIt->second, tr("Number Of Pixel Cells"), "");
                break;

            case VTK_QUAD:
                cellProp = new Property("Quads", elementsMapIt->second, tr("Number Of Quad Cells"), "");
                break;

            case VTK_TETRA :
                cellProp = new Property("Tetrahedra", elementsMapIt->second, tr("Number Of Tetrahedral Cells"), "");
                break;

            case VTK_VOXEL:
                cellProp = new Property("Voxels", elementsMapIt->second, tr("Number Of Voxel Cells"), "");
                break;

            case VTK_HEXAHEDRON :
                cellProp = new Property("Hexahedra", elementsMapIt->second, tr("Number Of Hexahedral Cells"), "");
                break;

            case VTK_WEDGE :
                cellProp = new Property("Wedges", elementsMapIt->second, tr("Number Of Wedge Cells"), "");
                break;

            case VTK_PYRAMID :
                cellProp = new Property("Pyramids", elementsMapIt->second, tr("Number Of Pyramid Cells"), "");
                break;

            case VTK_PENTAGONAL_PRISM:
                cellProp = new Property("Pentagonal Prisms", elementsMapIt->second, tr("Number Of Pentagonal Prism Cells"), "");
                break;

            case VTK_HEXAGONAL_PRISM:
                cellProp = new Property("Hexagonal Prisms", elementsMapIt->second, tr("Number Of Hexagonal Prism Cells"), "");
                break;

            default:
                cellProp = new Property("Others", elementsMapIt->second, tr("Number Of <i>Other Type Of Cells</i>. <br/>It can be quadratic isoparametric cells, Cubic isoparametric cells, <br/>convex group of points, higher order cells in parametric form, <br/>higher order cells (see VTKCellType enum for more information)"), "");
                break;
            }

            cellProp->setReadOnly(true);
            addProperty(cellProp);
        }
    }

    unsigned long memUsage = 0;

    if (getPointSet() != NULL)
        memUsage = getPointSet()->GetActualMemorySize();

    Property * memoryUsage = new Property("Size In Memory", QVariant(QString("%1").arg(memUsage)), tr("Actual size of the data in kilobytes. <br/>This number is valid only after the pipeline has updated. <br/>The memory size returned is guaranteed to be greater than or <br/>equal to the memory required to represent the data<br/> (e.g., extra space in arrays, etc. are not included in the return value)."), "Kb");
    memoryUsage->setReadOnly(true);
    addProperty(memoryUsage);
}

// -------------------- getSelections --------------------
QList< vtkSmartPointer< vtkSelectionNode > > & MeshComponent::getSelections() {
    return selectionList;
}

// -------------------- getNumberOfSelections --------------------
unsigned int MeshComponent::getNumberOfSelections() const {
    return selectionList.size();
}

// -------------------- getActiveSelection --------------------
vtkSmartPointer<vtkSelection> MeshComponent::getActiveSelection() const {
    return currentSelection;
}

// -------------------- getSelection --------------------
vtkSmartPointer< vtkSelectionNode > MeshComponent::getSelection(const QString & name) const {
    int indexOfSelection = getSelectionIndex(name);

    if (indexOfSelection >= 0 && indexOfSelection < selectionList.size())
        return getSelectionAt(indexOfSelection);
    else
        return NULL;
}

// -------------------- getSelectionAt --------------------
vtkSmartPointer< vtkSelectionNode > MeshComponent::getSelectionAt(unsigned int index) const {
    return selectionList.at(index);
}

// -------------------- getSelectionIndex --------------------
int MeshComponent::getSelectionIndex(const QString & name) const {
    QList< vtkSmartPointer< vtkSelectionNode > >::const_iterator it = selectionList.constBegin();
    int index = 0;
    bool found = false;

    while (it != selectionList.end() && !found) {
        if ((*it)->GetSelectionList() && !QString::compare(QString((*it)->GetSelectionList()->GetName()), name))
            found = true;
        else {
            index++;
            it++;
        }
    }

    if (found)
        return index;
    else
        return -1;
}

// -------------------- addSelection --------------------
int MeshComponent::addSelection(const QString & name, int fieldType, int contentType, vtkSmartPointer< vtkAbstractArray > array, MeshSelectionModel::InsertionPolicy policy) {
    int index = selectionModel->insertSelection(name, fieldType, contentType, array, policy);
    return index;
}

// -------------------- addToSelectedSelection --------------------
int MeshComponent::addToSelectedSelection(int fieldType, int contentType, vtkSmartPointer< vtkAbstractArray > array, MeshSelectionModel::InsertionPolicy policy) {
    // use the current index to handle the case of multiple selection
    int index = selectionView->selectionModel()->currentIndex().row();

    // TODO : for now, we don't care about the policy parameter and use the one of the combo box
    // but in the future, the policy should be selected by pressing modifiers keys
    policy = (MeshSelectionModel::InsertionPolicy) policyBox->itemData(policyBox->currentIndex()).toInt();

    // check if the current selection is selected (maybe it is not necessary ...)
    if (selectionView->selectionModel()->isRowSelected(index, QModelIndex()))
        index = addSelection(selectionList.at(index)->GetSelectionList()->GetName(), fieldType, contentType, array, policy);
    else
        index = addSelection("Picked Selection", fieldType, contentType, array, policy);

    // forced clearing even if the selected index stay the same
    // if not, display will not be updated
    selectionView->clearSelection();
    // select the added selection et set the current index
    selectionView->selectionModel()->setCurrentIndex(selectionModel->index(index, 0, QModelIndex()), QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows);

    return index;
}

// -------------------- changeSelectedSelection --------------------
// TODO : rename updateSelection
void MeshComponent::changeSelectedSelection(const QItemSelection & selected, const QItemSelection & deselected) {
    // remove all the selection nodes from the selection
    currentSelection->RemoveAllNodes();

    // do not use selected since it contains only the new selected selections
    // QModelIndexList items = selected.indexes();
    QModelIndex index;
    QModelIndexList items = selectionView->selectionModel()->selectedRows();

    // Select the PropertyExplorer 'Selection' tab for the currently selected component
    if (!Application::getSelectedComponents().isEmpty()) {
        Component * currentComponent = Application::getSelectedComponents().last();
        currentComponent->setIndexOfPropertyExplorerTab(1);
    }

    // add each selected selection nodes to the current selection
    foreach (index, items) {
        currentSelection->Union(selectionList.at(index.row()));
    }

    this->refresh();

}

// -------------------- removeSelectedSelection --------------------
void MeshComponent::removeSelectedSelections() {
    // TODO : handle multiple selection
    int index = selectionView->selectionModel()->currentIndex().row();

    if (selectionView->selectionModel()->isRowSelected(index, QModelIndex()))
        selectionModel->removeSelection(selectionList.at(index)->GetSelectionList()->GetName());
}

// -------------------- getDataArray --------------------
vtkSmartPointer<vtkDataArray> MeshComponent::getDataArray(MeshDataModel::FieldType fieldType, const QString & arrayName) {
    vtkSmartPointer< vtkFieldData > dataField;

    switch (fieldType) {
    case MeshDataModel::POINTS :
        dataField = getPointSet()->GetPointData();
        break;

    case MeshDataModel::CELLS :
        dataField = getPointSet()->GetCellData();
        break;

    case MeshDataModel::MESH :
        dataField = getPointSet()->GetFieldData();
        break;

    default:
        dataField = NULL;
        break;
    }

    return dataField->GetArray(arrayName.toStdString().c_str());
}

vtkSmartPointer<vtkDataArray> MeshComponent::getDataArray(MeshDataModel::FieldType fieldType, int index) {
    vtkSmartPointer< vtkFieldData > dataField;

    switch (fieldType) {
    case MeshDataModel::POINTS :
        dataField = getPointSet()->GetPointData();
        break;

    case MeshDataModel::CELLS :
        dataField = getPointSet()->GetCellData();
        break;

    case MeshDataModel::MESH :
        dataField = getPointSet()->GetFieldData();
        break;

    default:
        dataField = NULL;
        break;
    }

    return dataField->GetArray(index);
}

// -------------------- addDataArray --------------------
void MeshComponent::addDataArray(MeshDataModel::FieldType fieldType, const QString & name, vtkSmartPointer< vtkDataArray > data) {

    // get the field data set attributes
    vtkSmartPointer< vtkFieldData > dataField;

    switch (fieldType) {
    case MeshDataModel::POINTS :
        dataField = getPointSet()->GetPointData();
        break;

    case MeshDataModel::CELLS :
        dataField = getPointSet()->GetCellData();
        break;

    case MeshDataModel::MESH :
        dataField = getPointSet()->GetFieldData();
        break;

    default:
        dataField = NULL;
        break;
    }

    // attach the new data
    data->SetName(name.toStdString().c_str());
    dataField->AddArray(data);

    // set the data as active
    setActiveData(fieldType, name.toStdString().c_str());
}

// -------------------- removeDataArray --------------------
void MeshComponent::removeDataArray(MeshDataModel::FieldType fieldType, const QString & name) {
    vtkSmartPointer< vtkFieldData > dataField;

    switch (fieldType) {
    case MeshDataModel::POINTS :
        dataField = getPointSet()->GetPointData();
        break;

    case MeshDataModel::CELLS :
        dataField = getPointSet()->GetCellData();
        break;

    case MeshDataModel::MESH :
        dataField = getPointSet()->GetFieldData();
        break;

    default:
        dataField = NULL;
        break;
    }

    if (dataField->HasArray(name.toStdString().c_str())) {
        setActiveData(fieldType, NULL);
        dataField->RemoveArray(name.toStdString().c_str());
        this->removeProp(QString("data/" + MeshDataModel::getFieldNames()[fieldType] + "/" + name).toStdString().c_str());
    }

}

// -------------------- addPointData --------------------
void MeshComponent::addPointData(const QString & name, vtkSmartPointer< vtkDataArray > data) {
    addDataArray(MeshDataModel::POINTS, name, data);
}

// -------------------- addCellData --------------------
void MeshComponent::addCellData(const QString & name, vtkSmartPointer< vtkDataArray > data) {
    addDataArray(MeshDataModel::CELLS, name, data);
}

// -------------------- setActiveData --------------------
void camitk::MeshComponent::setActiveData(camitk::MeshDataModel::FieldType fieldType, const char * name) {
    vtkSmartPointer< vtkDataSetAttributes > dataAttr;
    bool meshDataType = false;

    if (fieldType == MeshDataModel::POINTS) {
        dataAttr = getPointSet()->GetPointData();
    } else {
        if (fieldType == MeshDataModel::CELLS) {
            dataAttr = getPointSet()->GetCellData();
        } else {
            meshDataType = true;
        }
    }

    if (!meshDataType) {
        // if name is NULL, remove all color scale
        if (name == NULL) {
            dataAttr->SetActiveScalars(NULL);
            // remove all color bar
            InteractiveViewer::get3DViewer()->setColorScaleTitle("");
            InteractiveViewer::get3DViewer()->setColorScale(false);
            refresh();
        } else {
            // set active scalar, vector or tensor
            switch (dataAttr->GetArray(name)->GetNumberOfComponents()) {
            case 1 : {
                dataAttr->SetActiveScalars(name);

                // show corresponding color bar
                double range[2];
                dataAttr->GetScalars()->GetRange(range); // range of the 1st component = unique component
                dataAttr->Update();
                //-- show the color scale in 3D
                InteractiveViewer::get3DViewer()->setColorScale(true);
                InteractiveViewer::get3DViewer()->setColorScaleTitle(name);
                InteractiveViewer::get3DViewer()->setColorScaleMinMax(range[0], range[1]);
            }
            break;

            case 3 :
                // TODO display the prop (visibility)
                dataAttr->SetActiveVectors(name);
                break;

            case 9 :
                dataAttr->SetActiveTensors(name);
                break;

            default:
                break;
            }
        }
    }

    // refresh the table
    dataModel->refresh();

}

// -------------------- getDataProp --------------------
vtkSmartPointer<vtkProp> MeshComponent::getDataProp(MeshDataModel::FieldType fieldType, const QString & arrayName) {
    QString fieldName;

    switch (fieldType) {
    case MeshDataModel::POINTS :
        fieldName = "points";
        break;

    case MeshDataModel::CELLS :
        fieldName = "cells";
        break;

    case MeshDataModel::MESH :
        fieldName = "mesh";
        break;

    default:
        fieldName = "";
        break;
    }

    QString dataName = ("data/" + fieldName + "/" + arrayName);

    vtkSmartPointer<vtkProp> prop = getProp(dataName);

    if (prop)
        return prop;

    vtkSmartPointer<vtkDataArray> array;

    if (getPointSet()->GetPointData()->HasArray(arrayName.toStdString().c_str())) {
        array = getPointSet()->GetPointData()->GetArray(arrayName.toStdString().c_str());

        switch (array->GetNumberOfComponents()) {
        case 3 : {
            vtkSmartPointer<vtkHedgeHog> hedgeHog = vtkSmartPointer<vtkHedgeHog>::New();
            vtkSmartPointer<vtkDataSetMapper> dataMapper = vtkSmartPointer< vtkDataSetMapper >::New();
            vtkSmartPointer<vtkActor> dataActor = vtkSmartPointer< vtkActor >::New();

            hedgeHog->SetInputConnection(myGeometry->getDataPort());
            hedgeHog->SetInputArrayToProcess(0, 0, 0, vtkDataObject::FIELD_ASSOCIATION_POINTS, array->GetName());
            dataMapper->SetInputConnection(hedgeHog->GetOutputPort());
            dataActor->SetMapper(dataMapper);
            dataActor->GetProperty()->SetColor(0, 1, 0);
            dataActor->VisibilityOff();

            addProp(dataName, dataActor);

            return dataActor;
            break;
        }

        case 9 : {
            vtkSmartPointer<vtkTensorGlyph> tensorGlyph = vtkSmartPointer<vtkTensorGlyph>::New();
            vtkSmartPointer<vtkDataSetMapper> dataMapper = vtkSmartPointer< vtkDataSetMapper >::New();
            vtkSmartPointer<vtkActor> dataActor = vtkSmartPointer< vtkActor >::New();
            vtkSmartPointer<vtkSphereSource> glyphSphere = vtkSmartPointer<vtkSphereSource>::New();

            glyphSphere->SetRadius(myGeometry->getBoundingRadius() / 50.0);
            tensorGlyph->SetInputConnection(myGeometry->getDataPort());
            tensorGlyph->SetInputArrayToProcess(0, 0, 0, vtkDataObject::FIELD_ASSOCIATION_POINTS, array->GetName());
            tensorGlyph->SetSourceConnection(glyphSphere->GetOutputPort());
            tensorGlyph->SetColorModeToEigenvalues();
            tensorGlyph->SetExtractEigenvalues(false);
            tensorGlyph->SetScaling(1);
            tensorGlyph->SetMaxScaleFactor(2);
            tensorGlyph->SetClampScaling(1);
            dataMapper->SetInputConnection(tensorGlyph->GetOutputPort());
            dataActor->SetMapper(dataMapper);
            dataActor->GetProperty()->SetColor(0, 1, 0);
            dataActor->VisibilityOff();

            addProp(dataName, dataActor);

            return dataActor;
            break;
        }

        default:
            return NULL;
            break;
        }
    }

    if (getPointSet()->GetCellData()->HasArray(arrayName.toStdString().c_str())) {
        array = getPointSet()->GetCellData()->GetArray(arrayName.toStdString().c_str());

        switch (array->GetNumberOfComponents()) {
        case 3 : {
            vtkSmartPointer<vtkCellCenters> cellCentersFilter = vtkSmartPointer<vtkCellCenters>::New();
            vtkSmartPointer<vtkHedgeHog> hedgeHog = vtkSmartPointer<vtkHedgeHog>::New();
            vtkSmartPointer<vtkDataSetMapper> dataMapper = vtkSmartPointer< vtkDataSetMapper >::New();
            vtkSmartPointer<vtkActor> dataActor = vtkSmartPointer< vtkActor >::New();

            cellCentersFilter->SetInputConnection(myGeometry->getDataPort());
            hedgeHog->SetInputConnection(cellCentersFilter->GetOutputPort());
            hedgeHog->SetInputArrayToProcess(0, 0, 0, vtkDataObject::FIELD_ASSOCIATION_CELLS, array->GetName());
            dataMapper->SetInputConnection(hedgeHog->GetOutputPort());
            dataActor->SetMapper(dataMapper);
            dataActor->GetProperty()->SetColor(0, 1, 0);
            dataActor->VisibilityOff();

            addProp(dataName, dataActor);

            return dataActor;
            break;
        }

        case 9 : {
            vtkSmartPointer<vtkCellCenters> cellCentersFilter = vtkSmartPointer<vtkCellCenters>::New();
            vtkSmartPointer<vtkTensorGlyph> tensorGlyph = vtkSmartPointer<vtkTensorGlyph>::New();
            vtkSmartPointer<vtkDataSetMapper > dataMapper = vtkSmartPointer< vtkDataSetMapper >::New();
            vtkSmartPointer<vtkActor> dataActor = vtkSmartPointer< vtkActor >::New();
            vtkSmartPointer<vtkSphereSource> glyphSphere = vtkSmartPointer<vtkSphereSource>::New();

            cellCentersFilter->SetInputConnection(myGeometry->getDataPort());
            glyphSphere->SetRadius(myGeometry->getBoundingRadius() / 50.0);
            tensorGlyph->SetInputConnection(cellCentersFilter->GetOutputPort());
            tensorGlyph->SetInputArrayToProcess(0, 0, 0, vtkDataObject::FIELD_ASSOCIATION_CELLS, array->GetName());
            tensorGlyph->SetSourceConnection(glyphSphere->GetOutputPort());
            tensorGlyph->SetColorModeToEigenvalues();
            tensorGlyph->SetExtractEigenvalues(false);
            tensorGlyph->SetScaling(1);
            tensorGlyph->SetMaxScaleFactor(2);
            tensorGlyph->SetClampScaling(1);
            dataMapper->SetInputConnection(tensorGlyph->GetOutputPort());
            dataActor->SetMapper(dataMapper);
            dataActor->GetProperty()->SetColor(0, 1, 0);
            dataActor->VisibilityOff();

            addProp(dataName, dataActor);

            return dataActor;
            break;
        }

        default:
            return NULL;
            break;
        }
    }

    return NULL;
}


// -------------------- removeSelectedData --------------------
void MeshComponent::removeSelectedData() {
    // TODO : handle multiple selection and cell/mesh data array deletion
    int index = dataView->selectionModel()->currentIndex().row();

    if (dataView->selectionModel()->isRowSelected(index, QModelIndex())) {
        vtkSmartPointer<vtkDataArray> arrayToDelete = getDataArray(MeshDataModel::POINTS, index);

        if (!arrayToDelete)
            CAMITK_WARNING("MeshComponent", "removeSelectedData", "Removal of data #" << index << " not implemented yet")
            else
                removeDataArray(MeshDataModel::POINTS, arrayToDelete->GetName());
    }
}

// -------------------- getNumberOfPropertyWidget --------------------
unsigned int MeshComponent::getNumberOfPropertyWidget() {
    return 3;
}

// -------------------- getPropertyWidgetAt --------------------
QWidget * MeshComponent::getPropertyWidgetAt(unsigned int i, QWidget * parent) {
    switch (i) {
    case 0 :
        return this->getPropertyWidget();
        break;

    case 1 :
        return selectionWidget;
        break;

    case 2 :
        return dataWidget;
        break;

    default:
        return NULL;
    }
}

// -------------------- getIcon --------------------
QPixmap MeshComponent::getIcon() {
    return QPixmap(":/cell");
}









/// TODO CAMITK_DEPRECATED.
/// This section list all the methods marked as deprecated. They are to be removed in CamiTK 4.0

// -------------------- getSelectionList --------------------
const QMap< QString, vtkSmartPointer< vtkSelectionNode > > & MeshComponent::getSelectionList() const {
    return selectionMap;
}

}








