/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
   \file cdw_mkudffs_regex.c

   Regex code for tools used to create UDF file system: mkudffs and
   its helpers.

   There is too little code to create separate *.c file for each tool,
   so I'm putting everything into this single file.
*/





#include <stdlib.h>
#include <string.h>

#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_regex_dispatch.h"
#include "cdw_debug.h"
#include "cdw_mkudffs_regex.h"
#include "cdw_config.h"
#include "cdw_processwin.h"
#include "cdw_logging.h"
#include "gettext.h"


extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];


extern cdw_task_t *thread_task;

/* stdout */
static int cdw_mkudffs_handle_rsync_progress(regex_t *regex, regmatch_t *matches);

/* stderr */
static int cdw_mkudffs_handle_sudo_password(regex_t *regex, regmatch_t *matches);
static int cdw_mkudffs_handle_segmentation_fault(regex_t *regex, regmatch_t *matches);
static int cdw_mkudffs_handle_mount_failure(regex_t *regex, regmatch_t *matches);
static int cdw_mkudffs_handle_umount_not_mounted(regex_t *regex, regmatch_t *matches);
static int cdw_mkudffs_handle_rsync_no_space_left(regex_t *regex, regmatch_t *matches);




static cdw_regex_t stdout_regex[] = {
	{ "rsync progress", 15001,
	  /*  "92,569,600  22%   44.15MB/s    0:00:07" */
	  "([0-9]+)%",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkudffs_handle_rsync_progress },

	{ "table guard", -1,
	  /* guard: debug_id = -1 */
	  "",
	  (regex_t *) NULL, (regmatch_t *) NULL, 0,
	  (cdw_regex_handler_t) NULL }
};





static cdw_regex_t stderr_regex[] = {
	{ "sudo password", 16001,
	  /*  "sudo: a password is required" */
	  "^sudo: a password",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkudffs_handle_sudo_password },

	{ "segmentation fault", 16002,
	  /*  "Segmentation fault" */
	  "^Segmentation fault",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkudffs_handle_segmentation_fault },

	{ "mount: wrong", 16003,
	  /* "mount: wrong fs type, bad option, bad superblock on /dev/loop0"... */
	  "^mount: wrong fs type, bad option",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkudffs_handle_mount_failure },

	{ "not mounted", 16004,
	  /*  "umount: /home/acerion/udf: not mounted" */
	  "^umount: (.+): not mounted",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkudffs_handle_umount_not_mounted },

	{ "mount: according", 16005,
	  /* "mount: according to mtab /home/acerion/u21.udf is already mounted on /home/acerion/udf as loop" */
	  "^mount: according to mtab",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkudffs_handle_mount_failure },

	{ "rsync: no space left", 16006,
	  /* "rsync: write failed on "/home/acerion/udf/big_file.txt": No space left on device (28)" */
	  "^rsync: (.+)o space left on device",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkudffs_handle_rsync_no_space_left },

	{ "table guard", -1,
	  /* guard: debug_id = -1 */
	  "",
	  (regex_t *) NULL, (regmatch_t *) NULL, 0,
	  (cdw_regex_handler_t) NULL }
};





/**
   \brief Do some initialization and call cdw_regex_prepare_regexes_table() for stdout regexes table
*/
void cdw_mkudffs_stdout_regex_prepare(void)
{
	cdw_rv_t crv = cdw_regex_prepare_regexes_table(stdout_regex);
	if (crv != CDW_OK) {
		cdw_vdm("ERROR: failed to prepare regexes table for stdout\n");
	}
	return;
}





/**
   \brief Do some initialization and call cdw_regex_execute_regexes_table()
   for stdout regexes table and stdout pipe buffer.
*/
void cdw_mkudffs_stdout_regex_execute(void)
{
	stdout_pipe_buffer[PIPE_BUFFER_SIZE] = '\0';
	cdw_regex_execute_regexes_table(stdout_regex, stdout_pipe_buffer);

	return;
}





/**
   \brief Call cdw_regex_clean_up_regexes_table() for stdout regexes table
*/
void cdw_mkudffs_stdout_regex_destroy(void)
{
	cdw_regex_clean_up_regexes_table(stdout_regex);
	return;
}





/**
   \brief Do some initialization and call cdw_regex_prepare_regexes_table() for stderr regexes table
*/
void cdw_mkudffs_stderr_regex_prepare(void)
{
	cdw_rv_t crv = cdw_regex_prepare_regexes_table(stderr_regex);
	if (crv != CDW_OK) {
		cdw_vdm("ERROR: failed to prepare regexes table for stderr\n");
	}
	return;
}





/**
   \brief Do some initialization and call cdw_regex_execute_regexes_table()
   for stderr regexes table and stderr pipe buffer.
*/
void cdw_mkudffs_stderr_regex_execute(void)
{
	stderr_pipe_buffer[PIPE_BUFFER_SIZE] = '\0';
	cdw_regex_execute_regexes_table(stderr_regex, stderr_pipe_buffer);

	return;
}





/**
   \brief Call cdw_regex_clean_up_regexes_table() for stderr regexes table
*/
void cdw_mkudffs_stderr_regex_destroy(void)
{
	cdw_regex_clean_up_regexes_table(stderr_regex);
	return;
}





/**
   sudo is prompting user for password

   cdw doesn't provide any way to pass password to sudo. User should
   know that "sudo mount" and "sudo umount" must be set up to be
   invoked without prompting for password.
*/
int cdw_mkudffs_handle_sudo_password(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* There is a regex defined in regex table, but we don't have
	   to extract any information from it.

	   Terminate child process (sudo prompting for password) by
	   sending Ctrl-D (EOT) to it. */

	thread_task->tool_status.sudo |= CDW_TOOL_STATUS_SUDO_PASSWORD_PROMPT;

	cdw_rv_t crv = cdw_thread_send_key_to_child_process(CDW_KEY_EOT);
	if (crv == CDW_OK) {
		return 0;
	} else {
		cdw_vdm ("ERROR: sending key to child process failed\n");
		return -1;
	}
}





/**
   Segmentation fault

   Most probably mkudffs, but we won't investigate this further.
*/
int cdw_mkudffs_handle_segmentation_fault(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.mkudffs_general |= CDW_TOOL_STATUS_MKUDFFS_GENERAL_SEGMENTATION_FAULT;

	return 0;
}





int cdw_mkudffs_handle_rsync_progress(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing progress information:
               1
	   "([0-9]+)%" */

	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	int done_percent = 0;

	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		if (i != 1) {
			continue;
		}
		char submatch[PIPE_BUFFER_SIZE + 1];
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 1) {
			done_percent = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, done percent = \"%s\" -> %d\n", i, submatch, done_percent);
		} else {
			cdw_vdm ("WARNING: unexpected submatch %d = %s\n", i, submatch);
		}
	}

	cdw_processwin_display_progress((long) done_percent, 100, (char *) NULL);

	cdw_processwin_wrefresh();

	return 0;
}





/* Failed to mount UDF file system under specified directory. Most
   probably problems with UDF file system in image file. */
int cdw_mkudffs_handle_mount_failure(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.mount |= CDW_TOOL_STATUS_MOUNT_FAILURE;

	return 0;
}





/* Trying to umount a directory that is not mounted. Most probably
   problems with creating UDF file system caused failure to mount, so
   there is nothing to umount. */
int cdw_mkudffs_handle_umount_not_mounted(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing "not mounted" message
	              1
	  "^umount: (.+): not mounted" */

	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	char submatch[PIPE_BUFFER_SIZE + 1];
	int len = cdw_regex_get_submatch(matches, 1, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", 1, len);
		return -1;
	}

	cdw_logging_write(_("ERROR: mount directory \"%s\" has not been mounted properly by cdw.\n"), submatch);

	thread_task->tool_status.umount |= CDW_TOOL_STATUS_UMOUNT_NOT_MOUNTED;

	return 0;
}





int cdw_mkudffs_handle_rsync_no_space_left(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.rsync |= CDW_TOOL_STATUS_RSYNC_NO_SPACE_LEFT;

	return 0;
}
