#ifndef _STREAM_H
#define _STREAM_H 1

// Representation of a single, muxed (we only really care about bytes/blocks) stream.
// Fed by Input, sent out by Server (to Client).

#include <stddef.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <deque>
#include <string>
#include <vector>

class StreamProto;
struct Client;

enum StreamStartSuitability {
	NOT_SUITABLE_FOR_STREAM_START,
	SUITABLE_FOR_STREAM_START,
};

struct Stream {
	// Must be in sync with StreamConfig::Encoding.
	enum Encoding { STREAM_ENCODING_RAW = 0, STREAM_ENCODING_METACUBE };

	Stream(const std::string &stream_id, size_t backlog_size, size_t prebuffering_bytes, Encoding encoding);
	~Stream();

	// Serialization/deserialization.
	Stream(const StreamProto &serialized, int data_fd);
	StreamProto serialize();

	// Changes the backlog size, restructuring the data as needed.
	void set_backlog_size(size_t new_size);

	// Mutex protecting <queued_data> and <queued_data_last_starting_point>.
	// Note that if you want to hold both this and the owning server's
	// <mutex> you will need to take <mutex> before this one.
	mutable pthread_mutex_t queued_data_mutex;

	std::string url;

	// The HTTP response header, without the trailing double newline.
	std::string http_header;

	// The video stream header (if any).
	std::string stream_header;

	// What encoding we apply to the outgoing data (usually raw, but can also
	// be Metacube, for reflecting to another Cubemap instance).
	Encoding encoding;

	// The stream data itself, stored in a circular buffer.
	//
	// We store our data in a file, so that we can send the data to the
	// kernel only once (with write()). We then use sendfile() for each
	// client, which effectively zero-copies it out of the kernel's buffer
	// cache. This is significantly more efficient than doing write() from
	// a userspace memory buffer, since the latter makes the kernel copy
	// the same data from userspace many times.
	int data_fd;

	// How many bytes <data_fd> can hold (the buffer size).
	size_t backlog_size;

	// How many bytes we need to have in the backlog before we start
	// sending (in practice, we will then send all of them at once,
	// and then start sending at the normal rate thereafter).
	// This is basically to force a buffer on the client, which can help
	// if the client expects us to be able to fill up the buffer much
	// faster than realtime (ie., it expects a static file).
	size_t prebuffering_bytes;

	// How many bytes this stream have received. Can very well be larger
	// than <backlog_size>, since the buffer wraps.
	size_t bytes_received;

	// A list of points in the stream that is suitable to start new clients at
	// (after having sent the header). Empty if no such point exists yet.
	std::deque<size_t> suitable_starting_points;
	
	// Clients that are in SENDING_DATA, but that we don't listen on,
	// because we currently don't have any data for them.
	// See put_client_to_sleep() and wake_up_all_clients().
	std::vector<Client *> sleeping_clients;

	// Clients that we recently got data for (when they were in
	// <sleeping_clients>).
	std::vector<Client *> to_process;

	// Maximum pacing rate for the stream.
	uint32_t pacing_rate;

	// Queued data, if any. Protected by <queued_data_mutex>.
	// The data pointers in the iovec are owned by us.
	struct DataElement {
		iovec data;
		StreamStartSuitability suitable_for_stream_start;
	};
	std::vector<DataElement> queued_data;

	// Put client to sleep, since there is no more data for it; we will on
	// longer listen on POLLOUT until we get more data. Also, it will be put
	// in the list of clients to wake up when we do.
	void put_client_to_sleep(Client *client);

	// Add more data to <queued_data>, adding Metacube headers if needed.
	// Does not take ownership of <data>.
	void add_data_deferred(const char *data, size_t bytes, StreamStartSuitability suitable_for_stream_start);

	// Add queued data to the stream, if any.
	// You should hold the owning Server's <mutex>.
	void process_queued_data();

private:
	Stream(const Stream& other);

	// Adds data directly to the stream file descriptor, without adding headers or
	// going through <queued_data>.
	// You should hold the owning Server's <mutex>, and probably call
	// remove_obsolete_starting_points() afterwards.
	void add_data_raw(const std::vector<DataElement> &data);

	// Remove points from <suitable_starting_points> that are no longer
	// in the backlog.
	// You should hold the owning Server's <mutex>.
	void remove_obsolete_starting_points();
};

#endif  // !defined(_STREAM_H)
