@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-file-mget: Download Multiple ECaccess Files at once
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use MIME::Base64;
use Term::Prompt;
use Number::Bytes::Human qw(format_bytes);

my %opt = ( progress => 0, force => 0, bufsize => 1048576, version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  progress
	  force
	  bufsize=i
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $targetLocalDirectory = pop(@ARGV);

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No source-ecaccess-file(s) specified!\n" ) if not(@ARGV) and not($targetLocalDirectory);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No target-local-directory specified!\n" ) if not(@ARGV);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Target-local-directory is not a directory!\n" ) if not( -d $targetLocalDirectory );

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# Process all source Files from the command-line
foreach (@ARGV) {
	my $sourceFiles = $_;

	# Get the list of Files to download
	my $files = $controlChannel->getDirList( $token, $sourceFiles );

	# Download each File
	foreach my $file ( $files->valueof('//getDirListResponse/return') ) {

		# Set source and target filenames
		my $source = $file->{domain} . "/" . $file->{name};
		my $target = $targetLocalDirectory . "/" . $file->{name};

		# Do we upload this file? (don't process directories and symbolic links)
		if (   not( substr( $file->{permissions}, 0, 1 ) eq 'd' )
			&& not( substr( $file->{permissions}, 0, 1 ) eq 'l' )
			&& $file->{size} >= '0'
			&& ( $opt{force} || &prompt( "y", "Download " . $source, "y/n", "y" ) ) )
		{
			print "Downloading " . $source . " -> " . $target . " (" . ( format_bytes( $file->{size} ) ) . ") ...\n";

			# Get the file handle and open the target file
			$handle = $controlChannel->getInputFileHandle( $token, $source, 0 )->result;
			open FILE, ">", $target or die "Error creating file: " . $target . "\n";

			# Progress bar
			my $progressbar;
			my $readCount = 0;
			my $readTotal = 0;
			if ( $opt{progress} && not( $^O =~ /^MSWin/ ) ) {
				eval "use Term::ProgressBar";
				$progressbar = Term::ProgressBar->new( { count => $file->{size}, remove => 1 } );
				$progressbar->update(0);
			}

			# Open the source file
			my $socket = $ecaccess->getFileInputStream($handle);

			# Read the file
			while (1) {
				$readCount = $socket->read_entity_body( $data, $opt{bufsize} );
				die "Error downloading file: " . $target . "\n" unless defined $readCount;
				last                                            unless $readCount;
				$readTotal += $readCount;
				print FILE $data;
				$progressbar->update($readTotal) if ( $opt{progress} && not( $^O =~ /^MSWin/ ) );
			}

			# Close the file handles
			$controlChannel->closeHandle($handle);
			close FILE;
		}
	}
}

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-file-mget - Download Multiple ECaccess Files at once

=head1 SYNOPSIS

B<ecaccess-file-mget -version|-help|-manual>

B<ecaccess-file-mget [-debug] [-progress] [-force] [-bufsize> I<length>B<]> I<source-ecaccess-file> B<[>I<...>B<]> I<target-local-directory>

=head1 DESCRIPTION

Allow downloading Multiple ECaccess Files at once in the specified I<target-local-directory>.

Each I<source-ecaccess-file> is in the form [domain:][/user-id/]path. Please read the "Shell commands -> File Management"
section of the "ecaccess" guide for more information on the ECaccess File System.

Please note this command is not recursive and will only download the plain-files in the I<source-ecaccess-file>
directory (sub-directories are ignored).

=head1 ARGUMENTS

=over 8

=item I<source-ecaccess-file> B<[>I<...>B<]>

The name(s) of the source ECaccess File(s).

=item I<target-local-directory>

The name of the target Local Directory.

=back

=head1 OPTIONS

=over 8

=item B<-progress>

Provide a progress meter on the standard terminal, allowing to monitor the file
transmission in real-time. The progress bar is removed from the terminal when
the transmission is done. This option is not supported and ignored on Windows
platforms.

=item B<-force>

Overrides the interactive mode and download each file without prompting.

=item B<-bufsize> I<length>

Specify the I<length> of the buffer (in bytes) which is used to download the files.
The larger the buffer the smaller the number of http/s requests. By default a
buffer of 1048576 bytes (1MB) is used.

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-file-mget> I<'home:/xyz/bin/*.bin'> I<$HOME/bin>

Download the I<*.bin> Files in the $HOME/bin directory of the user xyz. The Files are downloaded
in the local $HOME/bin directory.

=head1 SEE ALSO

B<ecaccess-file-delete>, B<ecaccess-file-get>, B<ecaccess-file-chmod>, B<ecaccess-file-modtime>, B<ecaccess-file-mput>,
B<ecaccess-file-rmdir>, B<ecaccess-file-copy>, B<ecaccess-file-dir>, B<ecaccess-file-mdelete>, B<ecaccess-file-mkdir>,
B<ecaccess-file-move>, B<ecaccess-file-put>, B<ecaccess-file-size> and B<ecaccess>.

=cut

__END__
:endofperl
