/***************************************************************************
                          chat.h  -  description
                             -------------------
    begin                : Wed Jan 15 22:41:32 CST 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CHAT_H
#define CHAT_H

#include "chatview.h"
#include "../contact/contactbase.h"
#include "../network/msnswitchboardconnection.h"

#include <QHash>
#include <QStringList>
#include <QTimer>


// Forward declarations
class ChatInformation;
class ChatMessage;
class ChatView;
class ChatWindow;
class ContactBase;
class KMessTest;
class MimeMessage;
class ChatHistoryWriter;

class QDir;



/**
 * Time before a typing message expires
 *
 * Expressed in seconds, this value determines for how long the chat window's status bar messages
 * "Contact is typing..." are displayed.
 */
#define CHAT_TYPING_EXPIRATION_TIME     5



/**
 * @brief Manage an active chat
 *
 * This class manages a chat. It handles the communication with the Switchboard connection;
 * and connects the Chat View signals and calls to make user interaction possible.
 *
 * It is meant to be inserted into a ChatWindow object, not to be used alone.
 *
 * @ingroup Chat
 */
class Chat : public ChatView
{
  Q_OBJECT

  friend class KMessTest;

  public:
    // The construtor
                       Chat( QWidget* parent = 0 );
    // The destructor
                      ~Chat();
    // Return a list of the contacts in the chat to be used as window caption
    const QString      getCaption();
    // Return the chat window which currently contains this Chat
    ChatWindow        *getChatWindow();
    // Return the list of contacts in the chat
    const QStringList  getParticipants() const;
    // Return the icon to use in the tab icon chat
    KIcon              getParticipantsTabIcon();
    // Return the list of previously sent messages
    QStringList        &getQuickRetypeList();
    // Return the date and time the chat has started
    QDateTime          getStartTime() const;
    // Return the list of contacts which are typing
    const QStringList  getTypingContacts() const;
    // Initialize the object
    bool               initialize( MsnSwitchboardConnection *switchboardConnection );
    // Return the switchboard connection used by the chat window.
    MsnSwitchboardConnection * getSwitchboardConnection() const;
    // Invite a contact to the chat
    void               inviteContacts( const QStringList &contacts );
    // Return whether or not the contact is in this chat. Check also if the chat is exclusive with the contact.
    bool               isContactInChat( const QString &handle, bool isExclusiveChatWithContact = false );
    // Return whether or not the chat is at its first incoming message
    bool               isChatFirstMessage();
    // The application is exiting
    void               queryExit();
    // The chat is closing
    void               queryClose();
    // Send a text message to the current chat
    void               sendChatMessage( const QString &message );
    // Send an ink drawing to the current chat
    void               sendInkMessage( InkFormat format, const QByteArray &inkData );
    // Send an "user is typing" message to the contacts
    void               sendTypingMessage();
    // Send an wink to the current chat
    void               sendWink( const MsnObject &message );
    // The user wants to block or unblock a contact.
    void               setContactBlocked( QString handle, bool isBlocked );
    // Enable or disable the parts of the window which allow user interaction
    void               setEnabled( bool isEnabled );
    // Start a chat
    void               startChat();
    // Show a wink received by a contact
    void               showWink( const QString &handle, const QString &filename, const QString &animationName );

  public slots:
    void               showMessage( const ChatMessage &message );
    // A message was received from a contact.
    void               receivedMessage(const ChatMessage &message);
    // Change the switchboard connection used by the chat window.
    void               setSwitchboardConnection( MsnSwitchboardConnection *newConnection = 0 );
    // Send a nudge
    void               slotSendNudge();
    // Send a file to a contact.
    void               startFileTransfer( QList<QUrl> fileList = QList<QUrl>() );

  private: // Private methods
    // Choose the contact to start an invitation with.
    const QString      chooseContact();
    // save the chat if needed.
    void               endChatLogging();

  private slots: // Private slots
    // A contact joined the chat
    void               contactJoined( ContactBase *contact, bool offlineContact = false );
    // A contact left the chat
    void               contactLeft  ( ContactBase *contact, bool isChatIdle );
    // A contact is typing
    void               contactTyping( ContactBase *contact = 0, bool forceExpiration = false );
    // A warning from the switchboard has been received, display it
    void               showWarning( MsnSwitchboardConnection::WarningType type, ContactBase* contact );
    // Display a message to notify of a contact status change
    void               slotContactChangedStatus();
    // Notify the user of a received nudge
    void               slotReceivedNudge( ContactBase* contact );
    // Notify that a message could not be sent by the switchboard
    void               slotSendingFailed( const QString &handle, const MimeMessage &message );

  private: // Private attributes
    // Whether or not the incoming message is the first incoming message
    bool               firstMessage_;
    // Whether or not the object was initialized
    bool               initialized_;
    // The last contacts known to have been in the chat. Kept for emergency, when there's no switchboard
    QStringList        lastKnownContacts_;
    // The last time an automessage has been sent
    QTime              lastSentAutoMessage_;
    // The xml chat log writer
    ChatHistoryWriter* logWriter_;
    // The object that connects to the switchboard server.
    MsnSwitchboardConnection *msnSwitchboardConnection_;
    // List of previously sent messages, to allow sending them again quickly
    QStringList        quickRetypeList_;
    // The date/time the chat started
    QDate              startDate_;
    QTime              startTime_;
    // The list of names of contacts which are currently typing
    QHash<QString,QString> typingContactsNames_;
    // The list of times when the contacts have started typing
    QHash<QString,QTime>   typingContactsTimes_;
    // A timer to update the typing information
    QTimer             typingTimer_;
    // Handle a command from ChatView (e.g. '/online')
    bool               handleCommand( QString command );

  signals:
    // Signal that some chat details (like the number of active contacts) have changed
    void               chatInfoChanged();
    // Signal that this chat is about to close.
    void               closing( Chat *chat );
    // Signal the presence of a new incoming chat message
    void               gotChatMessage( const ChatMessage &message, Chat *chat );
    // Signal that a typing message has been received
    void               gotTypingMessage( Chat *chat );
    // Signal that a nudge has been received
    void               gotNudge();
    // Signal that the ChatMaster should start a filetransfer.
    void               requestFileTransfer(const QString &handle, const QString &filename);
    // The user wants to add or remove a contact.
    void               contactAdded( QString handle, bool isAdded );
    // The user wants to allow a contact to see his/hers online status.
    void               contactAllowed( QString handle );
    // The user wants to block or unblock a contact.
    void               contactBlocked( QString handle, bool isBlocked );
    // The user wants to start private chat
    void               startPrivateChat( const QString &handle );
};

#endif
