#ifndef _PROTO_LFSR
#define _PROTO_LFSR

#include "protoDefs.h"     // for UINT32 and UINT8 types
#include "protoBitmask.h"  // for ProtoBitmask::GetWeight()

// This class implements Galois construction of
// of m-sequences that would be generated by a
// linear feedback shift register (LFSR)
class ProtoLFSR
{
    public:
            
        // These m-sequence generator polynomials are from
        // Forney[1970] via Proakis[1989], Table 5.2.5
        // Note the first set bit is "stage 1" of the 
        // shift register. In this Galois construction,
        // the output is collected from "stage m"
        enum Polynomial
        {                          // Connected shift register stages
            PN_NONE = 0x00000000,   // invalid polynomial
            PN3     = 0x00000003,   // 2 bits, stages 1,2
            PN7     = 0x00000005,   // 3 bits, stages 1,3
            PN15    = 0x00000009,   // 4 bits, stages 1,4
            PN31    = 0x00000012,   // 5 bits, stages 1,4
            PN63    = 0x00000021,   // 6 bits, stages 1,6
            PN127   = 0x00000041,   // 7 bits, stages 1,7
            PN255   = 0x0000008e,   // 8 bits, stages 1,5,6,7
            PN511   = 0x00000108,   // 9 bits, stages 1,6
            PN1023  = 0x00000204,   // 10 bits, stages 1,8
            PN2047  = 0x00000402,   // 11 bits, stages 1,10
            PN4095  = 0x00000829,   // 12 bits, stages 1,7,9,12
            PN8191  = 0x00001070,   // 13 bits, stages 1,7,8,9
            PN65535 = 0x00008805,   // 16 bits, stages 1,5,14,16
            PN24BIT = 0x00800043,   // 24 bits, stages 1,18,23,24
            PN32BIT = 0x80200003    // 32 bits, stages 1,11,31,32
        };    
        
        // The "reverse" parameter when set to "true" generates
        // the time-reversed version of the given PN sequence
        ProtoLFSR(Polynomial polynomial = PN2047,  
                  UINT32     initialState = 0xffffffff,
                  bool       reverse = false);
        
        Polynomial GetPolynomial() const
            {return (Polynomial)lfsr_poly;}
            
        UINT32 GetState() const
            {return (lfsr_state);}
        
        // Get lfsr size in bits
        unsigned int GetNumBits() const
            {return lfsr_bits;}
        
        // Get lfsr size in bytes (rounding up)
        unsigned int GetNumBytes() const
        {
            unsigned int n = lfsr_bits >> 3;
            return (0 != (lfsr_bits & 0x07)) ? n+1 : n;
        }
    
        UINT32 GetMask() const
            {return lfsr_mask;}
        
        void Reset(UINT32 initialState = 0xffffffff)
        {
            byte_mode = false;
            if (IsMirrored()) Mirror();
            lfsr_state = (initialState & lfsr_mask);
        }
        
        
        // This adjusts the "currsor" (i.e. position)
        // of the sequence.  I.e., GetNextBit() returns
        // the bit following the "cursor" and  GetPrevBit()
        // returns the bit preceding the "cursor"
        void Seek(int offset);
        
        // These methods can be used to "jog/shuttle" back and
        // forth through the generated sequence.
        // These return the next bit (or byte) of the sequence
        bool GetNextBit();
        UINT8 GetNextByte();
        
        // These return the bit (or byte) prior in the sequence
        // _before_ the last bit/byte produced by GetNextBit()
        // or GetNextByte(), respectively.
        bool GetPrevBit();
        UINT8 GetPrevByte();
        
        void FillBuffer(char* buffer, unsigned int buflen);
        
        
        // Here, buflen MUST be >= ((GetSize() + bitOffset) / 8)
        bool Sync(const char* buffer, unsigned int buflen, unsigned int bitOffset = 0);
        
        // Some static "helper" methods
        static unsigned int GetPolySize(Polynomial poly);
        static Polynomial GetPolynomial(unsigned int registerSize)
            {return POLYNOMIAL_LIST[registerSize &= 31];}
            
        static UINT32 PolynomialSearch(unsigned int registerSize);
        
        static unsigned char GetWeight(unsigned char c)
            {return ProtoBitmask::GetWeight(c);}
    
    private:
        static const ProtoLFSR::Polynomial POLYNOMIAL_LIST[33];
        void Shift(unsigned int count = 1);
        static UINT32 MirrorBits(UINT32 word, unsigned int numBits);
        void LoadBit(bool bit);   
        void Mirror();
        bool IsMirrored() const
            {return is_mirrored;}
        
        static bool GetBit(const char* buffer, unsigned int index)
            {return (0 != (buffer[index >> 3] & (0x80 >> (index & 0x07))));}
            
        UINT32          lfsr_poly; 
        UINT32          lfsr_state;
        unsigned int    lfsr_bits;
        UINT32          lfsr_mask;
        bool            is_mirrored;
        bool            byte_mode;
        
};  // end class ProtoLFSR

#endif // _PROTO_LFSR
