//
// Decompress.h
//
// $Id: //poco/1.3/Zip/include/Poco/Zip/Decompress.h#4 $
//
// Library: Zip
// Package: Zip
// Module:  Decompress
//
// Definition of the Decompress class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Zip_Decompress_INCLUDED
#define Zip_Decompress_INCLUDED


#include "Poco/Zip/Zip.h"
#include "Poco/Zip/ParseCallback.h"
#include "Poco/Zip/ZipArchive.h"
#include "Poco/Path.h"
#include "Poco/FIFOEvent.h"


namespace Poco {
namespace Zip {


class ZipArchive;


class Zip_API Decompress: public ParseCallback
	/// Decompress extracts files from zip files, can be used to extract single files or all files
{
public:
	typedef std::map<std::string, Poco::Path> ZipMapping;
		/// Maps key of FileInfo entries to their local decompressed representation
	Poco::FIFOEvent<std::pair<const ZipLocalFileHeader, const std::string> > EError;
		/// Thrown whenever an error is detected when handling a ZipLocalFileHeader entry. The string contains an error message
	Poco::FIFOEvent<std::pair<const ZipLocalFileHeader, const Poco::Path> > EOk;
		/// Thrown whenever a file was successfully decompressed

	Decompress(std::istream& in, const Poco::Path& outputDir, bool flattenDirs = false, bool keepIncompleteFiles = false);
		/// Creates the Decompress. Note that istream must be good and at the very beginning of the file!
		/// Calling decompressAllFiles will cause the stream to be in state failed once the zip file is processed.
		/// outputDir must be a directory. If it doesn't exist yet, it will be automatically created.
		/// If flattenDirs is set to true, the directory structure of the zip file is not recreated. 
		/// Instead, all files are extracted into one single directory.

	~Decompress();
		/// Destroys the Decompress.

	ZipArchive decompressAllFiles();
		/// Decompresses all files stored in the zip File. Can only be called once per Decompress object.
		/// Use mapping to retrieve the location of the decompressed files

	bool handleZipEntry(std::istream& zipStream, const ZipLocalFileHeader& hdr);

	const ZipMapping& mapping() const;
		/// A ZipMapping stores as key the full name of the ZipFileInfo/ZipLocalFileHeader and as value the decompressed file
		/// If for a ZipFileInfo no mapping exists, there was an error during decompression and the entry is considered to be corrupt

private:
	Decompress(const Decompress&);
	Decompress& operator=(const Decompress&);

	void onOk(const void*, std::pair<const ZipLocalFileHeader, const Poco::Path>& val);

private:
	std::istream& _in;
	Poco::Path    _outDir;
	bool          _flattenDirs;
	bool          _keepIncompleteFiles;
	ZipMapping    _mapping;
};


inline const Decompress::ZipMapping& Decompress::mapping() const
{
	return _mapping;
}


} } // namespace Poco::Zip


#endif // Zip_Decompress_INCLUDED
