#ifndef _RHEO_DOMAIN_H
#define _RHEO_DOMAIN_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:domain
NAME: @code{domain} - part of a finite element mesh
@cindex  mesh boundary
@clindex domain
DESCRIPTION:
  @noindent
  The @code{domain} class defines a container for a part of a
  finite element mesh.
  This describes the connectivity of edges or faces.
  This class is usefull for boundary condition setting.

DATE: 12 may 1997
METHODS: @domain
End:
*/
//
#include "rheolef/rheobase.h"
#include "rheolef/geo_element.h"
#include "rheolef/point.h"
#include "rheolef/Vector.h"

namespace rheolef { 

//<domain:
class domain : public Vector<geo_element> {
public:

// typedefs:

      typedef Vector<geo_element> sidelist_type;
      typedef Vector<geo_element>::size_type size_type;

// allocators/deallocators:

      domain(size_type sz = 0, const std::string& name = std::string());
      domain(const domain&);

// accessors:

      const std::string& name() const;
      size_type dimension() const;

// modifiers:

      void set_name(const std::string&);
      void set_dimension(size_type d);
      domain& operator=(const domain&);
      domain& operator += (const domain&);
      friend domain operator + (const domain&, const domain&);
      void resize(size_type n);
      void cat(const domain& d);
      template <class IteratorPair> 
	void set(IteratorPair p, size_type n, const std::string& name);
      template <class IteratorElement> 
	void set(IteratorElement p, size_type n, size_type dim, const std::string& name);

//  input/ouput:

      friend std::ostream& operator << (std::ostream& s, const domain& d);
      friend std::istream& operator >> (std::istream& s, domain& d);
      std::ostream& put_vtk (std::ostream& vtk, Vector<point>::const_iterator first_p,
		Vector<point>::const_iterator last_p) const;
      std::ostream& dump (std::ostream& s = std::cerr) const;
      void check() const;

// data:
protected:
      std::string   _name;
      size_type     _dim;

};
//>domain:


// ================== [ inlined ] ==================================================

inline
domain::domain(size_type n, const std::string& s)
 : Vector<geo_element>(n), 
   _name(s),
   _dim(0) 
{
}
inline
domain::domain(const domain& d)
 : Vector<geo_element>(d), 
   _name(d.name()),
   _dim(d._dim)
{
}
inline
const std::string&
domain::name() const 
{
    return _name;
}
inline
void
domain::set_name(const std::string& x)
{
    _name = x;
}
inline
domain::size_type
domain::dimension() const
{
    return _dim;
}
inline
void
domain::set_dimension(size_type d)
{
    _dim = d;
}
inline
void
domain::resize(size_type n)
{ 
    if (n > size()) {
        insert (end(), n - size(), geo_element());
    } else if (n < size ()) {
        erase (begin() + n, end());
    }
}
inline
domain& 
domain::operator = (const domain& d)
{
    Vector<geo_element>::operator = (d); 
    _name = d._name;
    _dim = d._dim;
    return *this;
}
inline
domain&
domain::operator += (const domain& d)
{
    cat(d);
    return *this;
}
inline
domain
operator + (const domain& d1, const domain& d2)
{
    domain d = d1;
    d += d2;
    return d;
}
template <class IteratorPair>
void
domain::set(IteratorPair p, size_type n, const std::string& name)
{
    _name = name;
    _dim = 1;
    resize(n);
    iterator last = end();
    for (iterator i = begin(); i != last; i++, p++) {
        (*i).set_name('e');
        (*i)[0] = (*p).first;
        (*i)[1] = (*p).second;
    }
}
template <class IteratorElement>
void
domain::set(IteratorElement p, size_type n, size_type dim, const std::string& name)
{
    _name = name;
    _dim  = dim;
    resize(n);
    iterator last = end();
    for (iterator i = begin(); i != last; i++, p++) {
	(*i).copy(*p);
    }
}
}// namespace rheolef
#endif // _RHEO_DOMAIN_H
