///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/basis_on_lattice.h"
using namespace rheolef;
// -----------------------------------------------------------------------
// basis evaluated on nodal lattice of Lagrange basis
// -----------------------------------------------------------------------
void 
basis_on_nodal_basis::_initialize (reference_element hat_K) const
{
    reference_element::variant_type K_type = hat_K.variant();
    std::vector<point> hat_node;
    _nb.hat_node (hat_K, hat_node);
    size_type nq = hat_node.size();
    _val[K_type].resize (nq);
    for (size_type q = 0; q < nq; q++) {
      _b.eval (hat_K, hat_node[q], _val[K_type][q]);
    }
    _initialized [K_type] = true;
}
basis_on_nodal_basis::const_iterator
basis_on_nodal_basis::begin (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _val[K_type][q].begin();
}
basis_on_nodal_basis::const_iterator
basis_on_nodal_basis::end (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _val[K_type][q].end();
}
// -----------------------------------------------------------------------
// basis evaluated on lattice of quadrature formulae
// -----------------------------------------------------------------------
void 
basis_on_quadrature::_initialize (reference_element hat_K) const
{
    reference_element::variant_type K_type = hat_K.variant();
    quadrature<Float>::const_iterator first = _p_quad -> begin(hat_K);
    size_type                  nq    = _p_quad -> size (hat_K);
    _val[K_type].resize (nq);
    for (size_type q = 0; q < nq; first++, q++) {
      const point& hat_node_q = (*first).x;
      _b.eval (hat_K, hat_node_q, _val[K_type][q]);
    }
    _initialized [K_type] = true;
}
basis_on_quadrature::const_iterator
basis_on_quadrature::begin (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _val[K_type][q].begin();
}
basis_on_quadrature::const_iterator
basis_on_quadrature::end (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _val[K_type][q].end();
}

void 
basis_on_quadrature::_grad_initialize (reference_element hat_K) const
{
    reference_element::variant_type K_type = hat_K.variant();
    quadrature<Float>::const_iterator first = _p_quad -> begin(hat_K);
    size_type                  nq    = _p_quad -> size (hat_K);
    _grad_val[K_type].resize (nq);
    for (size_type q = 0; q < nq; first++, q++) {
      const point& hat_node_q = (*first).x;
      _b.grad_eval (hat_K, hat_node_q, _grad_val[K_type][q]);
    }
    _grad_initialized [K_type] = true;
}
basis_on_quadrature::const_iterator_grad
basis_on_quadrature::begin_grad (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_grad_initialized [K_type]) _grad_initialize (hat_K);
    return _grad_val[K_type][q].begin();
}
basis_on_quadrature::const_iterator_grad
basis_on_quadrature::end_grad (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_grad_initialized [K_type]) _grad_initialize (hat_K);
    return _grad_val[K_type][q].end();
}void 

basis_on_quadrature::_hessian_initialize (reference_element hat_K) const
{
    reference_element::variant_type K_type = hat_K.variant();
    quadrature<Float>::const_iterator first = _p_quad -> begin(hat_K);
    size_type                  nq    = _p_quad -> size (hat_K);
    _hessian_val[K_type].resize (nq);
    for (size_type q = 0; q < nq; first++, q++) {
      const point& hat_node_q = (*first).x;
      _b.hessian_eval (hat_K, hat_node_q, _hessian_val[K_type][q]);
    }
    _hessian_initialized [K_type] = true;
}
basis_on_quadrature::const_iterator_hessian
basis_on_quadrature::begin_hessian (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_hessian_initialized [K_type]) _hessian_initialize (hat_K);
    return _hessian_val[K_type][q].begin();
}
basis_on_quadrature::const_iterator_hessian
basis_on_quadrature::end_hessian (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_hessian_initialized [K_type]) _hessian_initialize (hat_K);
    return _hessian_val[K_type][q].end();
}
