///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo
//
// authors:
//    	Pierre.Saramito@imag.fr
//
// date: 12 may 1997
//
//   added label_interface and build_subregion for updating meshes for 
//   multi-region problems. J.Etienne@damtp.cam.ac.uk, 19/9/2006
//
// ============================================================================
//  includes
// ============================================================================

#include "rheolef/georep.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
#include "rheolef/tiny_element.h"
#include "rheolef/tensor.h"
#include "geo-connectivity.h"
#include <string>
using namespace std;
namespace rheolef { 

// ============================================================================
//  utilities
// ============================================================================

template<class T>
class array2 {
public:
    typedef vector<int>::size_type size_type;
    array2() {}
    T& operator[](size_type i) { return _t[i]; }
    const T& operator[](size_type i) const { return _t[i]; }
protected:
    T _t[2];
};
template<class T>
class bidimmensional_array {
public:
    typedef vector<int>::size_type size_type;
    bidimmensional_array(size_type ni=0, size_type nj=0)
      : _nj(nj), _t(ni*nj) {}
    void resize (size_type ni, size_type nj) { _nj = nj; _t.resize(ni*nj); }
    T* operator[](size_type i) { return _t.begin() + i*_nj; }
    T* operator()(size_type i) { return operator[](i); }
    T& operator()(size_type i, size_type j) { return _t[i*_nj+j]; }
protected:
    size_type _nj;
    vector<T>    _t;
};
// ============================================================================
//  coordinate system mini-handler
// ============================================================================
string
georep::coordinate_system () const
{
    return fem_helper::coordinate_system_name (_coord_sys);
}
void
georep::set_coordinate_system (const string& name)
{
    _coord_sys = fem_helper::coordinate_system(name);
}
// ============================================================================
//  constructors
// ============================================================================

georep::georep(const georep& g)
 : Vector<geo_element>(g),
   _name(g._name),
   _number(g._number),
   _serial_number(g._serial_number),
   _zero_level_set_number(0),
   _dim(g._dim),
   _coord_sys(g._coord_sys),
   _x(g._x),
   _domlist(g._domlist),
   _xmin(g._xmin),
   _xmax(g._xmax),
   _save_double_precision(g._save_double_precision),
   _version(g._version),
   _tree(),
   _h(),
   _h1d(),
   _localizer_initialized(false),
   _boundary(g._boundary),
   _boundary_initialized(g._boundary_initialized)
{
    for (size_type i = 0; i < 4; i++)
  	_count_geo[i] = g._count_geo[i];
    for (size_type i = 0; i < reference_element::max_variant; i++)
  	_count_element[i] = g._count_element[i];
}
// ----------------------------------------------------------------------------
// cstor from file name
// ----------------------------------------------------------------------------

void
georep::reset_counters ()
{
   std::fill(_count_geo, _count_geo+4, 0);
   std::fill(_count_element, _count_element+reference_element::max_variant, 0);
}
georep::georep(const string& a_file_name, const string& coord_sys_name)
  : Vector<geo_element>(),
    _name(), 
    _number(numeric_limits<size_type>::max()),
    _serial_number(numeric_limits<size_type>::max()),
    _zero_level_set_number(0),
    _dim(0), 
    _coord_sys(fem_helper::coordinate_system(coord_sys_name)),
    _x(), 
    _domlist(), 
    _xmin(),
    _xmax(),
   _save_double_precision(false),
    _version(0),
    _tree(),
    _h(),
    _h1d(),
    _localizer_initialized(false),
    _boundary(),
    _boundary_initialized()
{
  reset_counters ();
  string file_name = a_file_name;
  irheostream is(file_name, "geo");
  if (!is.good()||(file_name.find("field")!=string::npos))
   {
   	warning_macro("\"" << file_name << "[.geo[.gz]]\" not found, trying for \"" << file_name << "[.[m]field[.gz]]\".");
	is.open(file_name, "mfield");
	if (!is.good()) is.open(file_name, "field");
	check_macro(is.good(), "\"" << file_name << "[.geo[.gz]]\" not found.");
	scatch(is,"\nfield");
	int version;
	size_type ndof;
	is >> version >> ndof >> file_name;
	is.open(file_name,"geo");
	check_macro(is.good(), "\"" << file_name << "[.geo[.gz]]\" not found.");
   }
  string root_name = delete_suffix (delete_suffix(file_name, "gz"), "geo");
  string base_name = get_basename (root_name);
  string::size_type i = base_name.find_last_of ('+');
  string::size_type j = base_name.find_last_of ('-');
  string::size_type l = base_name.length();
  if (j<i) j=l;
  if (i<l) 
   {
	string digits ="0123456789";
	string tmp =string(base_name, i+1, j-i-1);
	if (tmp.find_first_not_of(digits)==string::npos)
   		_number =atoi(tmp.c_str());
	else // it is not a valid version number
	 {
	 	warning_macro(tmp << " is not a valid mesh number in filename " << base_name);
		i=l; j=l;
	 }
	if (j<l)
	 {
	 	tmp =string(base_name, j+1, l-j-1);
		if (tmp.find_first_not_of(digits)==string::npos)
	  		_serial_number =atoi(tmp.c_str());
		else
	 	 {
		 	warning_macro(tmp << " is not a valid mesh serial number in filename " << base_name);
			i=l; j=l; _number=numeric_limits<size_type>::max();
	 	 }
	 }
	base_name = string(base_name, 0, i);
   }
  is >> setbasename(base_name) >> *this; // load data
  fem_helper::check_coord_sys_and_dimension (_coord_sys, _dim);
  _name = base_name;
}
// ----------------------------------------------------------------------------
// Cstructor from point list and former geo
// ----------------------------------------------------------------------------

georep::georep (const nodelist_type& p, const georep& g)
 : Vector<geo_element>(g),
   _name(g._name),
   _number(numeric_limits<size_type>::max()),
   _serial_number(numeric_limits<size_type>::max()),
   _zero_level_set_number(0),
   _dim(g._dim),
   _coord_sys(g._coord_sys),
   _x(p),
   _domlist(g._domlist),
   _xmin(),
   _xmax(),
   _save_double_precision(g._save_double_precision),
   _version(g._version),
   _tree(),
   _h(),
   _h1d(),
   _localizer_initialized(false),
   _boundary(g._boundary),
   _boundary_initialized(g._boundary_initialized)
{
  for (size_type j = 0; j < _dim; j++) {
    _xmin[j] =  numeric_limits<Float>::max();
    _xmax[j] = -numeric_limits<Float>::max();
  }
  for (size_type j = _dim+1; j < 3; j++) {
    _xmin [j] = _xmax [j] = 0;
  }
  nodelist_type::const_iterator iter_p = p.begin();
  nodelist_type::const_iterator last_p = p.end();

  while (iter_p != last_p) {
    const point& p = *iter_p++;
    for (size_type j = 0 ; j < _dim; j++) {
      _xmin[j] = min(p[j], _xmin[j]);
      _xmax[j] = max(p[j], _xmax[j]);
    }
  }

  if (g._number == numeric_limits<size_type>::max())
	_number = 1;
  else
	_number = g._number+1;
  _serial_number = numeric_limits<size_type>::max();
  
  for (size_type i = 0; i < 4; i++)
  	_count_geo[i] = g._count_geo[i];
  for (size_type i = 0; i < reference_element::max_variant; i++)
  	_count_element[i] = g._count_element[i];
}
// ----------------------------------------------------------------------------
//  cstor from domain
// ----------------------------------------------------------------------------

georep::georep(const georep& g, const domain& d)
  : Vector<geo_element>(d.size()),
    _name(d.name()),
    _number(numeric_limits<size_type>::max()),
    _serial_number(numeric_limits<size_type>::max()),
   _zero_level_set_number(0),
    _dim(g.dimension()),
    _coord_sys(g._coord_sys),
    _x(),
    _domlist(),
    _xmin(),
    _xmax(),
    _save_double_precision(g._save_double_precision),
    _version(2),
    _tree(),
    _h(),
    _h1d(),
    _localizer_initialized(false),
    _boundary(),
    _boundary_initialized()
{
  g.may_have_version_2();

  // step 0 : rename the mesh

  _name = d.name() + "_from_" + g.name() ;

  // step 1 : numbering of vertices

  vector<bool> node_on_d (g.n_node(), false);
  for (domain::const_iterator i = d.begin(), last_side = d.end(); i != last_side; i++) {

      const geo_element& S = *i;
      for (size_type j = 0; j < S.size(); j++)
	node_on_d [S[j]] = true ;
  }
  size_type nvertex = 0;
  vector<size_type> new_vertex_idx(g.n_node(), numeric_limits<size_type>::max());
  for (size_type i = 0; i < g.n_node(); i++) {
      if (node_on_d[i]) {
	new_vertex_idx[i] = nvertex++ ;
      }
  }
  for (size_type j = 0; j < _dim; j++) {
    _xmin[j] =  numeric_limits<Float>::max();
    _xmax[j] = -numeric_limits<Float>::max();
  }
  for (size_type j = _dim+1; j < 3; j++) {
    _xmin [j] = _xmax [j] = 0;
  }
  _x.resize(nvertex);
  size_type i_g_node = 0;
  iterator_node iter_node = begin_node() ;
  for (const_iterator_node iter_g_node = g.begin_node(); iter_g_node != g.end_node(); iter_g_node++, i_g_node++) {
      if (node_on_d[i_g_node]) {
	  *iter_node = *iter_g_node;
	  point& p = *iter_node;
          for (size_type j = 0 ; j < _dim; j++) {
              _xmin[j] = min(p[j], _xmin[j]);
              _xmax[j] = max(p[j], _xmax[j]);
          }
	  iter_node++;
      }
  }
  // step 2 : copy geo_element from "d" into geo_element of *this
  reset_counters ();
  _count_geo     [0] = n_node();
  _count_element [0] = n_node();
  resize(d.size());
  size_type K_idx = 0;
  iterator iter_elt = begin(); 
  for (domain::const_iterator iter_side = d.begin(), last_side = d.end(); 
	iter_side != last_side; 
	iter_side++, K_idx++, iter_elt++) {

      const geo_element& S = *iter_side;
      geo_element&       K = *iter_elt;
      K.set_variant(S.variant()) ;
      K.set_index (K_idx);
      for (size_type i = 0; i < K.size(); i++) {
  	  K[i] = new_vertex_idx [S[i]];
      }
      // update counters
      _count_geo    [K.dimension()]++;
      _count_element[K.variant()]++;
  }
  build_connectivity (*this);
}

// ============================================================================
// modifiers
// ============================================================================
void
georep::build_subregion(const domain& start_from, const domain& dont_cross, 
  	std::string name, std::string name_of_complement) 
{
    if (has_domain(name)) 
     {
        if (name_of_complement=="" || has_domain(name_of_complement))
     	 {
    	    warning_macro("Domain `" << name << "' already exists! Not overwriting.");
	    return;
     	 }
	else
	    error_macro("Domain `" << name << "' already exists, but not `"<<name_of_complement<<"'!");
     }
    else if (name_of_complement!="" && has_domain(name_of_complement))
	error_macro("Domain `" << name_of_complement << "' already exists, but not `"<<name<<"'!");
    bool update_complement=(name_of_complement!="");
    vector<set<size_type> > ball (n_node());
    build_point_to_element_sets (begin(), end(), ball.begin());
    // remember which are the vertices we don't want to cross
    set< size_type > no_way_vertices;
    domain::const_iterator iE=dont_cross.begin();
    domain::const_iterator eE=dont_cross.end();
    for ( ; iE!=eE; iE++) 
      for (size_type i=0; i<(*iE).size(); i++)
    	no_way_vertices.insert((*iE)[i]);
    // a list to remember the furthest we went in our exploration so far
    slist< size_type > frontier_dofs; // contains dofs
    // ...and a set to mark elemnts we've seen already
    set< size_type > elementa_cognita; // contains element indices
    // take a vertex in start_from to start with:
    domain::const_iterator i_start=start_from.begin();
    domain::const_iterator i_start_end=start_from.end();
    size_type v=(*i_start)[0];
    while (no_way_vertices.count(v)!=0)
    {
      i_start++;
      check_macro(i_start!=i_start_end, "No domain found between " << start_from.name() << " and " 
      	<< dont_cross.name() );
      v=(*i_start)[0];
    }
    frontier_dofs.push_front(v);
    do
    {
      v=*(frontier_dofs.begin());
      frontier_dofs.pop_front();
      set<size_type>::const_iterator iK=ball[v].begin();
      set<size_type>::const_iterator eK=ball[v].end();
      for (; iK!=eK; iK++)
      	if (elementa_cognita.count(*iK)==0)
	{
      	  elementa_cognita.insert(*iK);
	  for (size_type i=0; i<begin()[*iK].size(); i++)
	    if ( (begin()[*iK][i]!=v) && (no_way_vertices.count(begin()[*iK][i])==0) )
	      frontier_dofs.push_front(begin()[*iK][i]);
	}
    } while (!frontier_dofs.empty());
    // region has been explored, now fill in domain structures for it and its complement
    domain region, c_region;
    region.set_name(name);
    region.set_dimension(_dim);
    c_region.set_name(name_of_complement);
    c_region.set_dimension(_dim);
    const_iterator iK=begin();
    const_iterator eK=end();
    for ( ; iK!=eK; iK++)
    	if (elementa_cognita.count((*iK).index())==0) 
    	  c_region.push_back(*iK);
	else
    	  region.push_back(*iK);
    // insert regions in domain list
    _domlist.push_back(region);
    if (update_complement) _domlist.push_back(c_region);
}

// ============================================================================
// hazel initialization
// ============================================================================
void
georep::init_localizer (const domain& boundary, Float tolerance, int list_size) const
{
    if (_localizer_initialized)
        return;
    switch (dimension()) {
    case 1 : 
      _h1d.initialize (*this);
      break;
    case 2 : 
      _h.grow (begin(), end(), _x.begin(), _x.end(), 
	       boundary.begin(), boundary.end(),
	       size(), dimension(), xmin(), xmax(),
	       tolerance, list_size);
      break;
    case 3 :
      _tree.initialize (map_dimension(), xmin(), xmax(),
        n_vertex(), begin_node(), size(), begin());
      break;
    default : error_macro("localizer not defined in "  << dimension() << "D");
    }
    _localizer_initialized = true;
}
// ============================================================================
//  access
// ============================================================================

//
// index by domain
//
bool 
georep::has_domain (const string& domname) const
{
  domlist_type::const_iterator iter = _domlist.begin();
  domlist_type::const_iterator last = _domlist.end();
  while (iter != last) {
    const domain& dom = *iter;
    if (domname == dom.name())
      return true;
    ++iter;
  }
  return false;
}
const domain&
georep::operator[] (const string& domname) const
{
  domlist_type::const_iterator iter = _domlist.begin();
  domlist_type::const_iterator last = _domlist.end();
  while (iter != last) {
    const domain& dom = *iter;
    if (domname == dom.name())
      return dom;
    ++iter;
  }
  error_macro ("geo: undefined domain `" << domname << "'");
 
  return *iter;// for no warning at compile time
}
// ============================================================================
// upgrade
// ============================================================================

void
georep::upgrade()
{
    if (_version >= 2) {
	// already done
        return;
    }
    build_connectivity (*this);
    _version = 2;
}
// ============================================================================
// inquire
// ============================================================================

georep::size_type
georep::map_dimension() const
{
    for (size_type d = 3; d >= 0; d--) {
	if (_count_geo [d] != 0) {
	    return d;
	}
    }
    // has no elements...
    return _dim;
}

Float
georep::hmin() const 
{
    may_have_version_2();
    const_iterator_node p = begin_node();
    const_iterator iter_elt = begin();
    const_iterator last_elt = end();
    Float hm2 = numeric_limits<Float>::max();
    while (iter_elt != last_elt) {
        const geo_element& K = *iter_elt++;
        for (size_type e = 0; e < K.n_edge(); e++) {
	    size_type i1 = K.subgeo_vertex (1, e, 0);
	    size_type i2 = K.subgeo_vertex (1, e, 1);
	    hm2 = min(hm2, norm2(p[i1] - p[i2]));
        }
    }
    return ::sqrt(hm2);
}
Float
georep::hmax() const 
{
    may_have_version_2();
    const_iterator_node p = begin_node();
    const_iterator iter_elt = begin();
    const_iterator last_elt = end();
    Float hm2 = 0;
    while (iter_elt != last_elt) {
        const geo_element& K = *iter_elt++;
        for (size_type e = 0; e < K.n_edge(); e++) {
	    size_type i1 = K.subgeo_vertex (1, e, 0);
	    size_type i2 = K.subgeo_vertex (1, e, 1);
	    hm2 = max(hm2, norm2(p[i1] - p[i2]));
        }
    }
    return ::sqrt(hm2);
}
// =================================================================
// hatter and dehatter:
//  F : hat(K) --> K is a nonlinear transformation
// that depends upon the degree of the polynomial basis
// => depend on the "space" class
// => cannot be handled by the "geo" class
// => hatter and dehatter may move to the "piola" class (TODO)
// =================================================================
meshpoint 
georep::hatter (const point& x, size_type K_idx) const
{
    const geo_element& K = element(K_idx);
    switch(begin()[K_idx].variant()) {
        case geo_element::e: {
	    if (dimension()==1)
	     {
            	double a = begin_node()[begin()[K_idx][0]][0];
            	double b = begin_node()[begin()[K_idx][1]][0];
            	return meshpoint(K_idx, point((x[0] - a)/(b-a)));
	     }
	    else
	     {
            	point a = begin_node()[begin()[K_idx][0]];
            	point b = begin_node()[begin()[K_idx][1]];
		return meshpoint(K_idx, point( dot(x-a,b-a)/norm2(b-a) ));
	     }
	}	  
        case geo_element::t: {
	    check_macro(dimension()==2, "Triangles in 3D spaces not implemented");
            point a = begin_node()[begin()[K_idx][0]];
            point b = begin_node()[begin()[K_idx][1]];
            point c = begin_node()[begin()[K_idx][2]];
            point xx=x;
            Float t9 = 1/(-b[0]*c[1]+b[0]*a[1]+a[0]*c[1]+c[0]*b[1]-c[0]*a[1]-a[0]*b[1]);
            Float t11 = -a[0]+xx[0];
            Float t15 = -a[1]+xx[1];

            return meshpoint(K_idx, 
		point((-c[1]+a[1])*t9*t11-(-c[0]+a[0])*t9*t15,
                      (b[1]-a[1])*t9*t11-(b[0]-a[0])*t9*t15,
                       0));
        }
	// ICI
        case geo_element::T: {
            const point& a = vertex(K[0]);
            const point& b = vertex(K[1]);
            const point& c = vertex(K[2]);
            const point& d = vertex(K[3]);	
	    tensor A;
	    point ax;
	    for (size_t i = 0; i < 3; i++) {
	      ax[i]  = x[i]-a[i];
	      A(i,0) = b[i]-a[i];
	      A(i,1) = c[i]-a[i];
	      A(i,2) = d[i]-a[i];
	    }
	    tensor inv_A;
	    bool is_singular = ! invert_3x3 (A, inv_A);
            check_macro(!is_singular, "hatter: singular transformation in element: " << K.index());
	    point hat_x = inv_A*ax;
            return meshpoint(K_idx, hat_x); 
	}	  
        default : error_macro("hatter: element not yet implemented: " << K.name());
     }
}
point
georep::dehatter (const meshpoint& S, bool is_a_vector) const
{
    switch(begin()[S.element()].variant())
     {
        case geo_element::e: {
            double a = begin_node()[begin()[S.element()][0]][0];
            double b = begin_node()[begin()[S.element()][1]][0];
            return point(a + S.hat()[0]*(b-a));
	}	  
        case geo_element::t: {
            point a = begin_node()[begin()[S.element()][0]];
            point b = begin_node()[begin()[S.element()][1]];
            point c = begin_node()[begin()[S.element()][2]];
            return ((!is_a_vector)?a:point(0,0)) + (b-a)*S.hat()[0]
	                                         + (c-a)*S.hat()[1];
        }
        case geo_element::T: {
            point a = begin_node()[begin()[S.element()][0]];
            point b = begin_node()[begin()[S.element()][1]];
            point c = begin_node()[begin()[S.element()][2]];
            point d = begin_node()[begin()[S.element()][3]];
            return ((!is_a_vector)?a:point(0,0)) + (b-a)*S.hat()[0] 
                                                 + (c-a)*S.hat()[1]
                                                 + (d-a)*S.hat()[2];
        }
        default : error_macro("dehatter: element not yet implemented: "	
	      << begin()[S.element()].name());
     }
}
// ============================================================================
// localizer
// ============================================================================
bool 
georep::localize (const point& x, geo_element::size_type& K_idx) const
{
  switch (dimension()) {
    case 1 : 
      K_idx = _h1d.localize(*this, x[0]);
      return (K_idx != numeric_limits<hazel_1d::size_type>::max());
    case 2 : 
      return _h.localize(x, K_idx);
    case 3 :
warning_macro ("localize("<<x<<") = " << K_idx);
      K_idx = _tree.search_element_containing (begin(), x);
      return (K_idx != numeric_limits<size_t>::max());
    default : error_macro("localizer not defined in " << dimension() << "D");
  }
  return false;
}
void 
georep::localize_nearest (const point& x, point& y, geo_element::size_type& element) const
{
  switch (dimension()) {
    case 1 : 
	_h1d.localize_nearest (*this, x[0], y[0], element);
	return;
    case 2 : 
	return _h.localize_nearest (x, y, element);
    case 3 : // TODO
        element = _tree.find_close_element (begin(), x, y);
	return;
    default : error_macro("localizer/nearest not defined in " << dimension() << "D");
  }
}
bool 
georep::trace (const point& x0, const point& v, point& x, Float& t, size_type& element) const
{
  switch (dimension()) {
    case 1 : 
      return _h1d.trace (*this, x0[0], v[0], x[0], t, element);
    case 2 : 
      return _h.trace(x0, v, x, t, element);
    case 3 : // TODO
    default : error_macro("localizer/trace not defined in 3D");
  }
  return false;
}
// ============================================================================
//  basic input/output
// ============================================================================

// ----------------------------------------------------------------------------
//  output
// ----------------------------------------------------------------------------

void
georep::save() const
{
    orheostream file (name(), "geo");
    put(file);
}
ostream&
georep::dump (ostream& os) const
{
  os << "dump mesh" << endl
     << "name " << _name << endl
     << "version " << _version << endl
     << "dimension " << _dim << endl
     << "xmin " << _xmin << endl
     << "xmax " << _xmax << endl
    ;
  for (size_type i = 0; i < 4; i++)
     os << "count_geo[" << i << "] = " << _count_geo[i] << endl;
  for (size_type i = 0; i < reference_element::max_variant; i++)
     os << "count_element[" << i << "] = " << _count_element[i] << endl;

  size_type i = 0;
  os << "n_vertex " << _x.size() << endl;
  for (const_iterator_node iter_p = _x.begin(); iter_p != _x.end(); iter_p++, i++) {
      os << "x[" << i << "] = ";
      (*iter_p).put (os, _dim);
      os << endl ;
  }
  i = 0;
  os << "n_element " << size() << endl;
  for (const_iterator iter_e = begin(); iter_e != end(); iter_e++, i++) {
      os << "E[" << i << "]: ";
      (*iter_e).dump(os);
  }
  os << "n_domain " << _domlist.size() << endl;
  for (domlist_type::const_iterator i = _domlist.begin(); i != _domlist.end(); i++) {
	(*i).dump();
  }
  return os;
}
ostream& georep::put(ostream& s) const
{
  size_type map_d = map_dimension();
  //
  // put header
  //
  s << "#!geo" << endl 
    << "mesh" << endl ;

  s << version() << " " << dimension() << " " << n_node() << " " << size();
  if (version() >= 2) {
    if (dimension() >= 3) {
	if (map_d >= 3) {
		s << " " << n_face();
	} else {
		s << " 0";
	}
    }
    if (dimension() >= 2) {
	if (map_d >= 2) {
	    s << " " << n_edge();
	} else {
		s << " 0";
	}
    }
  }
  s << endl ;
  if (version() >= 3) {
    s << "coordinate_system " << coordinate_system() << endl;
  }
  s << endl ;
  //
  // put vertices
  //
  if (_save_double_precision) s << setprecision(numeric_limits<Float>::digits10);
  const_iterator_node iter_p = begin_node();
  const_iterator_node last_p = end_node();
  while (iter_p != last_p) {
      const point& p = *iter_p++;
      p.put (s, dimension());
      s << endl ;
  }
  s << endl ;
  //
  // put element
  //
  const_iterator iter_e = begin();
  const_iterator last_e = end();
  while (iter_e != last_e) {
      s << (*iter_e) << endl;
      iter_e++;
  }
  s << endl ;
  //
  // put faces
  //
  if (version() >= 2 && map_d >= 3) {
      vector<geo_element> f (n_face());
      const_iterator last_elt = end();
      for (const_iterator iter_elt = begin(); iter_elt != last_elt; iter_elt++) {
	  const geo_element& K = *iter_elt;
	  for (size_type i = 0; i < K.n_face(); i++) {
	      size_type idx = K.subgeo (2, i);
	      if (f [idx].variant() == reference_element::max_variant) {
		// not yet builded:
	        K.build_subgeo (2, i, f[idx]);
	      }
	  }
      }
      for (size_type j = 0; j < n_face(); j++) {
	s << f[j] << endl;
      }
      s << endl;
  }
  //
  // put edges
  //
  if (version() >= 2 && map_d >= 2) {
      vector<size_type> v1(n_edge());
      vector<size_type> v2(n_edge());
      const_iterator last_elt = end();
      for (const_iterator iter_elt = begin(); iter_elt != last_elt; iter_elt++) {
	  const geo_element& K = *iter_elt;
	  for (size_type i = 0; i < K.n_edge(); i++) {
	      v1 [K.edge(i)] = K.subgeo_vertex (1, i, 0);
	      v2 [K.edge(i)] = K.subgeo_vertex (1, i, 1);
	  }
      }
      for (size_type j = 0; j < n_edge(); j++) {
	s << v1[j] << " " << v2[j] << "\n" ;
      }
      s << endl ;
  }
  //
  // put domains
  //
  domlist_type::const_iterator iter = _domlist.begin();
  domlist_type::const_iterator last = _domlist.end();
  while (iter != last) s << endl << *iter++;
  s << endl ;

  return s ;
}
// =======================================================================
// check
// =======================================================================

void 
georep::check() const
{
warning_macro ("checking geo: elements...");
    if (_version <= 1) {
warning_macro ("checking geo done (version <= 1).");
 	return;
    }
    for (const_iterator i = begin(); i != end(); i++) {
        (*i).check();
        check(*i);
    }
warning_macro ("checking geo: domains...");
    domlist_type::const_iterator last = _domlist.end();
    for (domlist_type::const_iterator i = _domlist.begin(); i != last; i++) {
	(*i).check();
        for (domain::const_iterator j = (*i).begin(); j != (*i).end(); j++) {
            (*j).check();
            check(*j);
	}
    }
warning_macro ("checking geo done.");
}
void
georep::check (const geo_element& E) const
{
    for (size_type d = 0; d < E.dimension(); d++) {
        for (size_type i_subgeo = 0; i_subgeo < E.n_subgeo(d); i_subgeo++) {
	    for (size_type i = 0; i < E.subgeo_size(d, i_subgeo); i++) {
		size_type idx = E.subgeo_vertex(d, i_subgeo, i);
		check_macro (idx < _count_geo[d], d << "D subgeo index `" << idx 
		  << " out of range 0:" << _count_geo[d]);
	    }
	}
    }
}
// ----------------------------------------------------------------------------
//  input
// ----------------------------------------------------------------------------

istream&
get_nodes (
	istream& s, 
	georep::iterator_node iter_p, 
	georep::iterator_node last_p, 
	point& xmin, 
	point& xmax,
	georep::size_type dim)
{
  typedef georep::size_type size_type;
  for (size_type j = 0; j < dim; j++) {
    xmin[j] =  numeric_limits<Float>::max();
    xmax[j] = -numeric_limits<Float>::max();
  }
  for (size_type j = dim+1; j < 3; j++) {
    xmin [j] = xmax [j] = 0;
  }
  while (s.good() && iter_p != last_p) {
    point& p = *iter_p++;
    for (size_type j = 0 ; j < dim; j++) {
      s >> p[j] ; 
      xmin[j] = min(p[j], xmin[j]);
      xmax[j] = max(p[j], xmax[j]);
    }
    for (size_type j = dim; j < 3; j++)
      p [j] = 0;
  }
  return s;
}
istream&
georep::get_rheo(istream& is) 
{
  typedef georep::size_type size_type;
  bool verbose = iorheo::getverbose(is);
  
  if (!is.good()) error_macro("bad input stream for geo.");
  
  if (!scatch(is,"\nmesh"))
    error_macro("input stream does not contains a geo.");
  //
  // get header
  //
  size_type n_vert;
  size_type n_elt;
  size_type n_edg = 0;
  size_type n_fac = 0;

  is >> _version >> _dim >> n_vert >> n_elt;
  _x.resize(n_vert);
  resize(n_elt);
  if (_version == 1) {
      warning_macro ("obsolete format version 1");
  } else if (_version >= 2) {
      if (_dim >= 3) {
	is >> n_fac ;
      }
      if (_dim >= 2) {
	is >> n_edg ;
      }
  } else {
      fatal_macro("unknown mesh version " << _version);
  }
  if (_version >= 3) {
      if (!scatch(is,"\ncoordinate_system")) {
        error_macro("read geo version 3: `coordinate_system' expected");
      } else {
        string sys_coord;
	is >> sys_coord;
	set_coordinate_system (sys_coord);
      }
  }
  //
  // get coordinates
  //
  get_nodes (is, begin_node(), end_node(), _xmin, _xmax, _dim);
  if (!is.good()) return is;
  //
  // get elements
  //
  size_type idx = 0;
  reset_counters ();
  _count_geo     [0] = n_node();
  _count_element [0] = n_node();
  georep::iterator last_elt = end();
  for (georep::iterator i = begin(); is.good() && i != last_elt; i++, idx++) {

      is >> (*i);
      (*i).set_index (idx);
      _count_geo     [(*i).dimension()]++;
      _count_element [(*i).variant()]++;
  }
  size_type map_d = map_dimension();

  vector<set<size_type> > ball (n_node());
  if (_version >= 2 && map_d > 0) {
  
      // TODO: _count_element [type]++

      ball.resize(n_node());
      build_point_to_element_sets (begin(), end(), ball.begin());
      if (map_d >= 3) { // get faces
	  load_subgeo_numbering (begin(), ball.begin(), 2, n_fac, is,
		_count_geo, _count_element);
      }
      load_subgeo_numbering (begin(), ball.begin(), 1, n_edg, is,
		_count_geo, _count_element);
  }
  //
  // get domains
  //
  while (is.good()) {
      domain d;
      if (!(is >> d)) {
	  break;
      }
      if (_version >= 2 && d.dimension() > 0 && size() > 0) {
	propagate_subgeo_numbering (d.begin(), d.end(), begin(), ball.begin());
      }
      _domlist.push_back(d);
  }
  return is;
}
istream&
operator >> (istream& s, georep& m) 
{
    string basename = iorheo::getbasename(s);
    if (basename.length() != 0) {
	m._name = basename;
    } else {
	m._name = "unamed";
	warning_macro ("unamed input mesh has been named `" << m._name << "'");
    }
    m.reset_counters ();
    iorheo::flag_type fmt = iorheo::flags(s) & iorheo::format_field;
    if      (fmt [iorheo::bamg])        { return m.get_bamg (s); }
    else if (fmt [iorheo::grummp])      { return m.get_grummp (s); }
    else if (fmt [iorheo::gmsh])        { return m.get_gmsh (s); }
    else if (fmt [iorheo::mmg3d])       { return m.get_mmg3d (s); }
    else if (fmt [iorheo::tetgen])      { return m.get_tetgen (s); }
    else if (fmt [iorheo::qmg])         { return m.get_qmg (s); }
    else if (fmt [iorheo::cemagref])    { return m.get_cemagref (s); }
    else if (fmt [iorheo::vtkpolydata]) { return m.get_vtk_polydata (s); }
    else                                { return m.get_rheo (s); }
}
void
georep::may_have_version_2() const
{
    if (version() < 2) {
	error_macro("geo version 2 requiered (see: geo -upgrade): " 
		<< _version << " founded");
    }
}
// ----------------------------------------------------------------------------
//  domain modifiers
// ----------------------------------------------------------------------------
void
georep::erase_domain (const string& name) 
{
  for (domlist_type::iterator q = _domlist.begin(); q != _domlist.end(); q++) {
    if (name == (*q).name()) {
      _domlist.erase(q);
      return;
    }
  }
}
void
georep::insert_domain (const domain& d)
{
      _domlist.push_back(d);
}
// ============================================================
//  construction of jump-interface domain data:
// ============================================================
void
georep::jump_interface(const domain& interface, const domain& region, 
	map<size_type,tiny_vector<size_type> >& special_elements, 
	map<size_type,size_type>& node_global_to_interface) const
{
  check_macro((_dim==2)&&(interface.dimension()==1), "Only 2D implemented");
  domain::const_iterator edge_i =interface.begin();
  domain::const_iterator end_edges =interface.end();
  set<size_type> n_tmp;
  for ( ;  edge_i!=end_edges ; edge_i++)
   {
	n_tmp.insert((*edge_i)[0]);
	n_tmp.insert((*edge_i)[1]);
	//node_pair* pair0 = new node_pair((*edge_i)[0], 0); 
	//if (!(node_global_to_interface.insert(*pair0).second)) (*pair0).~pair();
	//node_pair* pair1 = new node_pair((*edge_i)[1], 0); 
	//if (!(node_global_to_interface.insert(*pair1).second)) (*pair1).~pair();
   }
  set<size_type>::iterator n_tmp_i =n_tmp.begin();
  set<size_type>::iterator end_n_tmp =n_tmp.end();
  for ( size_type j=0 ; n_tmp_i != end_n_tmp ; n_tmp_i++, j++ )
   {
    	//node_global_to_interface.insert(node_global_to_interface.end(),node_pair((*n_tmp_i),j));
    	node_global_to_interface.insert(make_pair((*n_tmp_i),j));
   }
  map<size_type,size_type>::iterator end_nodes =node_global_to_interface.end();
  map<size_type,tiny_vector<size_type> >::iterator special_K;
  domain::const_iterator ele_i =region.begin();
  domain::const_iterator end_elements =region.end();
  tiny_vector<size_type> *l_conn;
  for ( ; ele_i != end_elements ; ele_i++)
   {
  	bool already_inserted=false;
   	for (size_type j=0; j <(*ele_i).size(); j++)
	 {
		map<size_type,size_type>::const_iterator node
			=node_global_to_interface.find((*ele_i)[j]);
	 	if (node!=end_nodes)
		 {
			if (!already_inserted)
			 {
			 	already_inserted=true;
				l_conn= new tiny_vector<size_type>((*ele_i).size());
				(*l_conn).fill(numeric_limits<size_type>::max());
				special_K = special_elements.insert(
					make_pair((*ele_i).index(),*l_conn)).first;
			 }
			(*special_K).second[j] = (*node).second;
		 }
	 }
   }
}
// ============================================================================
//  comparator
// ============================================================================

// small cpu-time equality check
// we base on name changing convention
// when mesh change ! 
bool
georep::operator == (const georep& y) const
{
    // rought connectivity comparison
    if (size() != y.size()) return false;
    if (n_node() != y.n_node()) return false;
    if (n_edge() != y.n_edge()) return false;
    if (n_face() != y.n_face()) return false;
    if (n_domain() != y.n_domain()) return false;

    // when only node coords change, we change the name !
    if (name() != y.name()) return false;

    return true;
}

// ============================================================================
//   Boundary builders 
// ============================================================================
const domain&
georep::boundary () const
{
    if (_boundary_initialized) return _boundary;
    _boundary_initialized = true;
    size_type d = dimension();
    _boundary.set_name("boundary");
    vector<set<size_type> > ball (n_node());
    build_point_to_element_sets (begin(), end(), ball.begin());
    build_boundary (begin(), end(), ball.begin(), d-1, _boundary);
    return _boundary;
}
void
georep::label_interface(const domain& dom1, const domain& dom2, const string& name)
{    
    // for 2D domains only
    check_macro(dom1.dimension()==2 && dom2.dimension()==2, "only 2D domains supported"); 
    geo_element K;
    K.set_name('e');
    domain interface;
    interface.set_dimension(1); 
    interface.set_name(name);
    // first step: sort vertices of larger domain in a set
    domain smaller, larger;
    if (dom1.size()<dom2.size()) { smaller=dom1; larger=dom2; }
    	else { smaller=dom2; larger=dom1; }
    set< size_type > larger_vertices;
    domain::const_iterator iK=larger.begin();
    domain::const_iterator eK=larger.end();
    for ( ; iK!=eK; iK++) 
      for (size_type i=0; i<(*iK).size(); i++)
    	larger_vertices.insert((*iK)[i]);
    // second step: find smaller domain vertices which are within set and build interface
    iK=smaller.begin();
    eK=smaller.end();
    for ( ; iK!=eK; iK++) 
      for (size_type i=0, j=(*iK).size(); i<(*iK).size(); i++, j=i-1)
      	if ( (larger_vertices.count((*iK)[j])>0)
	   && (larger_vertices.count((*iK)[i])>0) ) 
	     {
	     	K[0]=(*iK)[j]; K[1]=(*iK)[i];
	   	interface.push_back(K);
	     }
    _domlist.push_back(interface);
}
Float
georep::measure (const geo_element& K) const
{
  georep::const_iterator_vertex p = begin_vertex();
  switch (K.variant()) {
    case geo_element::p:
      return 0;
    case geo_element::e:
      switch (dimension()) {
       case 1: return       fabs(p[K[1]][0] - p[K[0]][0]);
       case 2: return ::sqrt (sqr(p[K[1]][0] - p[K[0]][0])
                             +sqr(p[K[1]][1] - p[K[0]][1]));
       default: return dist(p[K[1]], p[K[0]]);
      }
    case geo_element::t:
      switch (dimension()) {
       case 2:  return fabs(vect2d( p[K[1]] -p[K[0]], p[K[2]] -p[K[0]] )/2.0);
       default: return norm(vect  ( p[K[1]] -p[K[0]], p[K[2]] -p[K[0]] )/2.0);
      }
    case geo_element::q: {
      switch (dimension()) {
       case 2: {
        Float area1 = fabs(vect2d( p[K[1]] -p[K[0]], p[K[2]] -p[K[0]] )/2);
        Float area2 = fabs(vect2d( p[K[2]] -p[K[0]], p[K[3]] -p[K[0]] )/2);
        return area1 + area2;
       }
       default: {
        Float area1 = norm(vect( p[K[1]] -p[K[0]], p[K[2]] -p[K[0]] )/2.0);
        Float area2 = norm(vect( p[K[2]] -p[K[0]], p[K[3]] -p[K[0]] )/2.0);
        return area1 + area2;
       }
      }
    }
    case geo_element::T:
       return fabs(mixt(p[K[1]]-p[K[0]], p[K[2]]-p[K[0]], p[K[3]]-p[K[0]])/6);
    default:
      error_macro("area: element " << K.name() << " not yet supported yet.");
      return 0;
  }
}
// compute normal to a side of an element
point
georep::normal (const geo_element& K, georep::size_type side) const
{
  switch (K.variant()) {
    case reference_element::e:  { // works also in 2d and 3d
      point t = vertex(K[1]) - vertex(K[0]);
      t = t/norm(t);
      if (side == 1) return t; else return -t;
    }
    case reference_element::t:
    case reference_element::q: { // works also in 3d
      point t = vertex(K[(side+1)%3]) - vertex(K[side]);
      return point(t[1],-t[0])/norm(t);
    }
    // TODO: T, P, H...
    default:
      error_macro("normal to element not yet implemented for element type `"
	<< K.variant() << "'");
  }
}
// compute normal to a facet: triangle in 3D, etc
point
georep::normal (const geo_element& S) const
{
  switch (S.variant()) {
    case reference_element::e:  { // 2d space
      point t = vertex(S[1]) - vertex(S[0]);
      t = t/norm(t);
      return point (-t[1], t[0]);
    }
    case reference_element::t:
    case reference_element::q: { // 3d space
      const point& a = vertex(S[0]);
      const point& b = vertex(S[1]);
      const point& c = vertex(S[2]);
      point n = vect (b-a, c-a);
      return n/norm(n);
    }
    default:
      error_macro("normal to facet not yet implemented for element type `"
	<< S.variant() << "'");
  }
}
}// namespace rheolef
