use core:io;
use core:net;
use crypto;

/**
 * Connection to a client on the server.
 */
class ServerConn {
	init(NetStream stream, Database db) {
		init {
			socket = stream;
			input = ObjIStream(stream.input);
			output = ObjOStream(stream.output);
			db = db;
		}
	}

	init(NetStream stream, IStream input, OStream output, Database db) {
		init {
			socket = stream;
			input = ObjIStream(input);
			output = ObjOStream(output);
			db = db;
		}
	}

	private Database db;
	private NetStream socket;
	private ObjIStream input;
	private ObjOStream output;

	// Authenticated user ID.
	private Int myUserId;

	// Client key for this user (for logout mainly).
	private Str myClientId;

	// Get user ID.
	public Int userId() { myUserId; }

	// Get the client ID.
	public Str clientId() { myClientId; }

	// Run the communication with this client.
	void run() {
		try {
			if (!authenticate()) {
				socket.close();
				return;
			}

			while (handleMessage())
				;

		} catch (Exception e) {
			print("Unhandled exception in the server:");
			print(e.toS);
		}

		socket.close();
	}

	private Bool handleMessage() {
		try {
			Request request = Request:read(input);
			if (response = request.execute(this, db)) {
				send(response);
				return true;
			} else {
				// No reply, that means we should disconnect.
				return false;
			}

		} catch (EndOfStream eos) {
			print("End of stream. Exiting.");
			return false;
		} catch (ServerError e) {
			// This is considered continuable.
			error(e.message);
			return true;
		} catch (Exception e) {
			print("Exception in the server: " + e.toS);
			error("Exception in the server: " + e.message);
			return false;
		}

		true;
	}

	// Authenticate the user on the other side.
	private Bool authenticate() {
		Request request = Request:read(input);
		if (request as AuthRequest) {
			print("Checking for user: ${request.key}...");
			if (user = db.findUser(request.key)) {
				myUserId = user.id;
				myClientId = request.key;
				send(AuthResponse(user.name));
				return true;
			} else {
				// Note: This is fine since we know that the key is URL-safe.
				Url url = httpsUrl("storm-lang.org") / "progvis" / ("login.php?id=" + request.key);
				send(AuthLoginResponse(url));
				return false;
			}
		} else {
			error("The first message must be an authentication request!");
			return false;
		}
	}

	// Send an error message back.
	private void error(Str message) {
		send(ErrorResponse(message));
	}

	// Send a message to the client.
	private void send(Response response) {
		response.write(output);
		output.flush();
	}
}

// Run the server.
void serverMain() on Network {
	Url cert = cwdUrl / "cert.pem";
	Url key = cwdUrl / "key.pem";
	if (cert.exists & key.exists) {
		var c = Certificate:loadPEM(cert);
		var k = c.loadKeyPEM(key);

		print("Launching SSL server...");
		sslServerMain(ServerContext(k));
	} else {
		print("Launching non-SSL server...");
		plainServerMain();
	}

	print("Server terminated!");
	// TODO: Restart?
}

// Non-SSL version.
private void plainServerMain() on Network {
	unless (listener = listen(serverPort))
		throw ServerError("Failed to listen to ${serverPort}!");

	Database db;

	while (stream = listener.accept()) {
		ServerConn client(stream, db);
		(spawn client.run()).detach();
	}
}

// SSL version.
private void sslServerMain(ServerContext context) on Network {
	unless (listener = listen(serverPort))
		throw ServerError("Failed to listen to ${serverPort}!");

	Database db;

	while (stream = listener.accept()) {
		(spawn sslClient(stream, context, db)).detach();
	}
}

// Perform SSL handshake for a single client, and run it.
private void sslClient(NetStream stream, ServerContext context, Database db) on Network {
	try {
		var session = context.connect(stream);

		ServerConn client(stream, session.input, session.output, db);
		client.run();
	} catch (Exception e) {
		print("Error from client: ${e.message}");
	}
}

// Run a server that accepts a single client.
void simpleServer() on Network {
	unless (listener = listen(serverPort))
		throw ServerError("Failed to listen to ${serverPort}!");

	Database db;
	if (stream = listener.accept()) {
		ServerConn client(stream, db);
		client.run();
	}
}
